<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * @author    Presta.Site
 * @copyright 2015 Presta.Site
 * @license   LICENSE.txt
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

@ini_set('max_execution_time', 14400);

class PrestaBackup extends Module
{
    private $html;
    private $exec_avail = false;

    public $settings_prefix = 'PSB_';

    // db settings:
    public $backup_db;
    public $backup_all;
    public $drop_table;
    public $max_db_backups;
    public $db_method;

    //files settings:
    public $backup_files;
    public $files_arch_type;
    public $files_ignore_cache;
    public $max_files_backups;

    // custom settings:
    public $backup_suffix = '-psbckp';
    public $exclude_paths_cache = array(
        '/cache/smarty/cache/',
        '/cache/cachefs/',
        '/cache/smarty/compile/',
    );
    public $exclude_paths_always = array();
    public $exclude_paths_custom;
    public $restore_errors = array();
    public $conf = array();

    public function __construct()
    {
        $this->name = 'prestabackup';
        $this->tab = 'administration';
        $this->version = '1.1.2';
        $this->author = 'Presta.Site';
        $this->bootstrap = true;
        $this->ps_versions_compliancy = array('min' => '1.5', 'max' => '1.7.99.99');

        parent::__construct();

        $this->loadSettings();

        $this->displayName = $this->l('PrestaBackup');
        $this->description = $this->l('Backup module');
    }

    public function install()
    {
        if (!parent::install()
            or !$this->registerHook('displayBackOfficeHeader')
        ) {
            return false;
        }

        $this->checkCreateTableComments();

        //default values:
        foreach ($this->getSettings() as $item) {
            if ($item['type'] == 'html') {
                continue;
            }
            if (Configuration::get($this->settings_prefix.$item['name']) === false) {
                Configuration::updateValue($this->settings_prefix.$item['name'], $item['default']);
            }
        }

        return true;
    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }

        return true;
    }

    public function getBackupDirs()
    {
        $backup_dirs = array(
            'root' => _PS_MODULE_DIR_.$this->name.'/backup/',
            'db' => _PS_MODULE_DIR_.$this->name.'/backup/db/',
            'files' => _PS_MODULE_DIR_.$this->name.'/backup/files/',
        );

        return $backup_dirs;
    }

    public function hookDisplayBackOfficeHeader($params)
    {
        if (Tools::getValue('configure') != $this->name) {
            return false;
        }

        $this->context->controller->addJquery();
        $this->context->controller->addCSS($this->_path.'views/css/admin.css');
        $this->context->controller->addJS($this->_path.'views/js/admin.js');
        $this->context->controller->addJqueryPlugin('fancybox');

        $this->context->smarty->assign(array(
            'psbackup_path' => $this->_path,
        ));

        return $this->display(__FILE__, 'backofficeheader.tpl');
    }

    public function getContent()
    {
        $this->html = '';
        $this->checkCreateTableComments();
        $this->html .= $this->postProcess();
        if (Tools::getValue('psbp_conf')) {
            foreach (Tools::getValue('psbp_conf') as $id) {
                $this->html .= $this->displayConfirmation($this->getConfirmationMsg($id));
            }
        }
        $this->html .= (Configuration::get($this->settings_prefix.'HIDE_WARNING') ? '' : $this->renderWarning());
        $this->html .= $this->checkMaxExecutionTime();
        $this->html .= $this->renderBackupForm();
        $this->html .= $this->renderBackupsTable();
        $this->html .= $this->renderSettingsForm();
        $this->html .= $this->renderAdditionalData();
        $this->html .= (Configuration::get($this->settings_prefix.'HIDE_WARNING') ? $this->renderWarning() : '');

        return $this->html;
    }

    protected function postProcess()
    {
        $html = '';
        $this->exec_avail = $this->checkExecAvail();
        $errors = array();
        $settings_updated = false;
        if (Tools::isSubmit('submitModule')) {
            //saving settings:
            $settings = $this->getSettings();
            foreach ($settings as $item) {
                if ($item['type'] == 'html' or (isset($item['lang']) && $item['lang'] == true)) {
                    continue;
                }
                if (Tools::isSubmit($item['name'])) {
                    Configuration::updateValue(
                        $this->settings_prefix.$item['name'],
                        Tools::getValue($item['name']),
                        true
                    );
                    $settings_updated = true;
                }
            }

            //update lang fields:
            $languages = Language::getLanguages();
            foreach ($settings as $item) {
                if ($item['type'] == 'html') {
                    continue;
                }
                $lang_value = array();
                foreach ($languages as $lang) {
                    if (Tools::isSubmit($item['name'].'_'.$lang['id_lang'])) {
                        $lang_value[$lang['id_lang']] = Tools::getValue($item['name'].'_'.$lang['id_lang']);
                        $settings_updated = true;
                    }
                }
                if (sizeof($lang_value)) {
                    Configuration::updateValue($this->settings_prefix.$item['name'], $lang_value, true);
                }
            }

            if (Tools::strtoupper(Tools::substr(PHP_OS, 0, 3)) === 'WIN' or !$this->exec_avail) {
                Configuration::updateValue($this->settings_prefix.'DB_METHOD', 'default');
                Configuration::updateValue($this->settings_prefix.'FILES_ARCH_TYPE', 'zip');
            }
        }

        if ($settings_updated) {
            $this->conf[] = 1;
        }

        if (Tools::isSubmit('doBackup') and $this->isBackupDirsWritable()) {
            if (Tools::getValue('run_options_unlock') == 'unlock') {
                Configuration::updateValue($this->settings_prefix.'IN_PROGRESS', 0);
            }
            if (Tools::getValue('run_options_db') == 'db') {
                $this->backupDB();
            }
            if (Tools::getValue('run_options_files') == 'files') {
                $this->backupFiles();
            }
        }

        $this->loadSettings();

        if (!$this->isBackupDirsWritable()) {
            $errors[] = sprintf(
                $this->l('Module directory and its subdirectories must be writable: %s'),
                str_replace(_PS_ROOT_DIR_, '', _PS_MODULE_DIR_.$this->name.'/')
            );
        }

        if (sizeof($this->conf) && !sizeof($errors)) {
            // delete old backups in according of submitted settings
            $this->deleteOldBackups('db');
            $this->deleteOldBackups('files');

            $token = Tools::getAdminTokenLite('AdminModules');
            $confirmation = '';
            foreach ($this->conf as $conf) {
                $confirmation .= '&psbp_conf[]='.(int)$conf;
            }
            $redirect_url = 'index.php?tab=AdminModules&configure=' . $this->name . '&token=' . $token . $confirmation;
            Tools::redirectAdmin($redirect_url);
        } elseif (sizeof($errors)) {
            foreach ($errors as $err) {
                $html .= $this->displayError($err);
            }
        }
        
        return $html;
    }

    /**
     * Generates settings form
     * @return string, generated html
     */
    protected function renderSettingsForm()
    {
        $fields_form = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Settings'),
                    'icon' => 'icon-cogs',
                ),
                'input' => array_merge($this->getDbSettings(), $this->getFilesSettings()),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );
        if ($this->getPSVersion() == 1.5) {
            $fields_form['form']['submit']['class'] = 'button';
        }

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ?
            Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $this->fields_form = array();
        if ($this->getPSVersion() == 1.5) {
            $helper->module = $this;
        }

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false).
            '&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => array(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        foreach ($this->getSettings() as $item) {
            if ($item['type'] == 'html') {
                continue;
            }
            $helper->tpl_vars['fields_value'][$item['name']] = Configuration::get($this->settings_prefix.$item['name']);
        }

        return $helper->generateForm(array($fields_form));
    }

    /**
     * Generates form that allow you to create backup
     * @return string, generated html
     */
    protected function renderBackupForm()
    {
        $this->context->smarty->assign(array(
            'psv' => $this->getPSVersion(),
            'action' => $this->context->link->getAdminLink('AdminModules', false).
                '&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules')
        ));

        return $this->context->smarty->fetch($this->local_path . 'views/templates/admin/create_backup_form.tpl');
    }

    /**
     * Db backup settings
     * @return array
     */
    public function getDbSettings()
    {
        $settings = array(
            array(
                'type' => 'html',
                'name' => '',
                'label' => '',
                'html_content' => '<b>'.$this->l('Database backup settings:').'</b>',
            ),
            array(
                'type' => $this->getPSVersion() == 1.5 ? 'radio' : 'switch',
                'name' => 'BACKUP_ALL',
                'label' => $this->l('Ignore statistics tables:'),
                'hint' => _DB_PREFIX_.'connections, '._DB_PREFIX_.'connections_page, '._DB_PREFIX_
                    .'connections_source, '._DB_PREFIX_.'guest, '._DB_PREFIX_.'statssearch',
                'class' => 't',
                'values' => array(
                    array(
                        'id' => 'backup_all_on',
                        'value' => 1,
                        'label' => $this->l('Yes'),
                    ),
                    array(
                        'id' => 'backup_all_off',
                        'value' => 0,
                        'label' => $this->l('No'),
                    ),
                ),
                'default' => 0,
            ),
            array(
                'type' => $this->getPSVersion() == 1.5 ? 'radio' : 'switch',
                'name' => 'DROP_TABLE',
                'label' => $this->l('Drop existing tables during import:'),
                'hint' => $this->l('If enabled, the backup script will drop your tables prior to restoring data.
                                    (ie. "DROP TABLE IF EXISTS")'),
                'class' => 't',
                'values' => array(
                    array(
                        'id' => 'drop_on',
                        'value' => 1,
                        'label' => $this->l('Yes'),
                    ),
                    array(
                        'id' => 'drop_off',
                        'value' => 0,
                        'label' => $this->l('No'),
                    ),
                ),
                'default' => 1,
            ),
            array(
                'type' => 'text',
                'name' => 'MAX_DB_BACKUPS',
                'label' => $this->l('Max number of DB backups:'),
                'hint' => $this->l('Maximum number of backups stored on your server.
                                    Older backups will be deleted if this value will be exceeded.'),
                'class' => 'fixed-width-sm',
                'default' => 3,
            ),
            'db_method' => array(
                'type' => 'select',
                'name' => 'DB_METHOD',
                'label' => $this->l('Method'),
                'hint' => $this->l('Try other option if you have any problems.'),
                'class' => 't',
                'options' => array(
                    'query' => array(
                        array(
                            'id_option' => 'default',
                            'name' => 'default',
                        ),
                    ),
                    'id' => 'id_option',
                    'name' => 'name',
                ),
                'default' => 'default',
            ),
            array(
                'type' => 'html',
                'name' => '',
                'label' => '',
                'html_content' => '<hr/>',
            ),
        );

        // disable "exec tar.gz" backup on Windows or if it is not avail
        if (Tools::strtoupper(Tools::substr(PHP_OS, 0, 3)) !== 'WIN' and $this->exec_avail) {
            $settings['db_method']['options']['query'][] = array(
                'id_option' => 'mysqldump',
                'name' => $this->l('mysqldump'),
            );
        }

        if ($this->getPSVersion() < 1.6) {
            foreach ($settings as &$item) {
                $desc = isset($item['desc']) ? $item['desc'] : '';
                $hint = isset($item['hint']) ? $item['hint'].'<br/>' : '';
                $item['desc'] = $hint.$desc;
                $item['hint'] = '';
            }

        }

        return $settings;
    }

    /**
     * Files backup settings
     * @return array
     */
    public function getFilesSettings()
    {
        $settings = array(
            array(
                'type' => 'html',
                'name' => '',
                'label' => '',
                'html_content' => '<b>'.$this->l('Files backup settings:').'</b>',
            ),
            'files_arch_type' => array(
                'type' => 'select',
                'name' => 'FILES_ARCH_TYPE',
                'label' => $this->l('Archive format'),
                'hint' => $this->l('Try other option if you have any problems.'),
                'class' => 't',
                'options' => array(
                    'query' => array(
                        array(
                            'id_option' => 'zip',
                            'name' => 'zip',
                        ),
                    ),
                    'id' => 'id_option',
                    'name' => 'name',
                ),
                'default' => 'zip',
            ),
            array(
                'type' => $this->getPSVersion() == 1.5 ? 'radio' : 'switch',
                'name' => 'FILES_IGNORE_CACHE',
                'label' => $this->l('Ignore cache:'),
                'hint' => $this->l('Skip shop cache files when adding to backup archive.'),
                'class' => 't',
                'values' => array(
                    array(
                        'id' => 'cache_ignore_on',
                        'value' => 1,
                        'label' => $this->l('Yes'),
                    ),
                    array(
                        'id' => 'cache_ignore_off',
                        'value' => 0,
                        'label' => $this->l('No'),
                    ),
                ),
                'default' => 1,
            ),
            array(
                'type' => 'text',
                'name' => 'MAX_FILES_BACKUPS',
                'label' => $this->l('Max number of files backups:'),
                'hint' => $this->l('Maximum number of backups stored on your server.
                                    Older backups will be deleted if this value will be exceeded.'),
                'class' => 'fixed-width-sm',
                'default' => 2,
            ),
            array(
                'type' => 'textarea',
                'name' => 'EXCLUDE_PATHS_CUSTOM',
                'label' => $this->l('Exclude directories, one per line:'),
                'hint' => $this->l('Exclude some directories from backup.'),
                'desc' => $this->l('Example: /img/'),
                'default' => '',
            ),
        );

        // disable "exec tar.gz" backup on Windows or if it is not avail
        if (Tools::strtoupper(Tools::substr(PHP_OS, 0, 3)) !== 'WIN' and $this->exec_avail) {
            $settings['files_arch_type']['options']['query'][] = array(
                'id_option' => 'targzExec',
                'name' => 'tar.gz ',
            );
        }

        if ($this->getPSVersion() < 1.6) {
            foreach ($settings as &$item) {
                $desc = isset($item['desc']) ? $item['desc'] : '';
                $hint = isset($item['hint']) ? $item['hint'].'<br/>' : '';
                $item['desc'] = $hint.$desc;
                $item['hint'] = '';
            }

        }

        return $settings;
    }

    /**
     * Return all module settings
     * @return array
     */
    public function getSettings()
    {
        return array_merge(
            $this->getDbSettings(),
            $this->getFilesSettings()
        );
    }

    /**
     * Load settings from db to module variables
     */
    protected function loadSettings()
    {
        foreach ($this->getSettings() as $item) {
            if ($item['type'] == 'html') {
                continue;
            }
            $name = Tools::strtolower($item['name']);
            $this->$name = Configuration::get($this->settings_prefix.$item['name']);
        }
    }

    /**
     * Return PrestaShop version
     * @param bool|false $without_dots
     * @return float
     */
    protected function getPSVersion($without_dots = false)
    {
        $ps_version = _PS_VERSION_;
        $ps_version = Tools::substr($ps_version, 0, 3);

        if ($without_dots) {
            $ps_version = str_replace('.', '', $ps_version);
        }

        return (float)$ps_version;
    }

    /**
     * Starts backup creation
     * @return string
     */
    public function runBackup()
    {
        $last_start = Configuration::get($this->settings_prefix.'LAST_START');
        $days_diff = 0;
        if ($last_start) {
            $days_diff = ((time() - $last_start)) / 86400;
        }

        //create backup if backup process is not in progress or if previous backup process seems to be broken
        if (!Configuration::get($this->settings_prefix.'IN_PROGRESS') or $days_diff >= 1) {
            Configuration::updateValue($this->settings_prefix.'IN_PROGRESS', 1);
            Configuration::updateValue($this->settings_prefix.'LAST_START', time());

            if ($this->backup_db) {
                $this->backupDB();
            }
            if ($this->backup_files) {
                $this->backupFiles();
            }

            Configuration::updateValue($this->settings_prefix.'IN_PROGRESS', 0);
        } else {
            $this->html .= $this->displayError('Backup is already in progress.');
        }

        return $this->html;
    }

    /**
     * @return bool
     */
    public function backupDB()
    {
        if (!$this->isBackupDirsWritable()) {
            return false;
        }

        //create db backup
        require_once(_PS_MODULE_DIR_.$this->name.'/classes/PSDbBackup.php');
        $prestaShopBackup = new PSDbBackup();

        if ($this->db_method == 'mysqldump') {
            $db_user = _DB_USER_;
            $db_passwd = _DB_PASSWD_;
            $db_name = _DB_NAME_;

            $rand = dechex(mt_rand(0, min(0xffffffff, mt_getrandmax())));
            $date = time();
            $file_path = _PS_MODULE_DIR_.$this->name.'/backup/db/'.$date.'-db-'.$rand.$this->backup_suffix.'.sql.gz';

            $cmd = "mysqldump -u$db_user -p'$db_passwd' --single-transaction --routines --triggers $db_name | gzip > ".
                escapeshellarg($file_path);
            exec($cmd);

            $prestaShopBackup->id = $file_path;
        } else {
            $prestaShopBackup->add();
            $file_path = $prestaShopBackup->id;
        }

        //display result
        if ($file_path and file_exists($file_path)) {
            // Comment:
            if (Tools::getValue('run_options_comment')) {
                $filename = basename($file_path);
                $this->addComment($filename, Tools::getValue('run_options_comment'));
            }

            $this->conf[] = 2;

            $this->deleteOldBackups('db');
        } else {
            $this->html .= $this->displayError($this->l('Db backup failed.'));
        }

        return ($file_path and file_exists($file_path));
    }

    /**
     * @return bool
     */
    public function backupFiles()
    {
        if (!$this->isBackupDirsWritable()) {
            return false;
        }

        $rand = dechex(mt_rand(0, min(0xffffffff, mt_getrandmax())));
        $date = time();
        $fileName = _PS_MODULE_DIR_.$this->name.'/backup/files/'.$date.'-files-'.$rand.$this->backup_suffix;

        if ($this->files_arch_type == 'tar.gz') {
            $result_file = $this->targz(_PS_ROOT_DIR_, $fileName);
        } elseif ($this->files_arch_type == 'zip') {
            $result_file = $this->zip(_PS_ROOT_DIR_, $fileName);
        } elseif ($this->files_arch_type == 'targzExec') {
            $result_file = $this->targzExec(_PS_ROOT_DIR_, $fileName);
        }

        if (isset($result_file) and file_exists($result_file)) {
            // Add comment
            if (Tools::getValue('run_options_comment')) {
                $filename = basename($result_file);
                $this->addComment($filename, Tools::getValue('run_options_comment'));
            }

            $this->conf[] = 4;

            $this->deleteOldBackups('files');

            return true;
        } else {
            $this->html .= $this->displayError($this->l('Files backup failed.'));

            return false;
        }
    }

    /**
     * @param bool|false $from_root
     * @return string
     */
    protected function getBackupDirectoryPath($from_root = false)
    {
        if (!$from_root) {
            return _PS_MODULE_DIR_.$this->name.'/backup/';
        } else {
            return '/modules/'.$this->name.'/backup/';
        }
    }

    /**
     * @return bool
     */
    protected function isBackupDirsWritable()
    {
        foreach ($this->getBackupDirs() as $dir) {
            if (!is_writable($dir)) {
                return false;
            }
        }
        if (!is_writable(_PS_MODULE_DIR_.$this->name)) {
            return false;
        }

        return true;
    }

    /**
     * Return MIME for db backup file
     * @param $file_path
     * @return string
     */
    public function getMIME($file_path)
    {
        $mime = 'text/plain';

        $mimes = array(
            'bz2' => 'application/x-bzip2',
            'gz' => 'application/x-compressed',
            'zip' => 'application/zip',
        );
        $extension = pathinfo($file_path, PATHINFO_EXTENSION);

        if (isset($mimes[$extension])) {
            $mime = $mimes[$extension];
        }

        return $mime;
    }

    /**
     * Checks if backup mail template exists
     * @return bool
     */
    protected function checkMailTemplate()
    {
        $iso = Language::getIsoById(Configuration::get('PS_LANG_DEFAULT'));
        $mails_dir = _PS_MODULE_DIR_.$this->name.'/mails/';

        //if template exists, return true
        if (file_exists($mails_dir.$iso.'/backup.txt') and file_exists($mails_dir.$iso.'/backup.html')) {
            return true;
        } else {
            //if template cannot be created return false
            if (!is_writable($mails_dir)) {
                $this->html .= $this->displayError(
                    sprintf(
                        $this->l('Email template directory is not writable. Please check for write access (%s)'),
                        str_replace(_PS_ROOT_DIR_, '', $mails_dir)
                    )
                );

                return false;
            }

            if (!is_dir($mails_dir.$iso)) {
                $mkdir = mkdir($mails_dir.$iso);
                if (!$mkdir) {
                    $this->html .= $this->displayError(
                        $this->l('Cannot create email template directory. Please check for write access.')
                    );

                    return false;
                }
            }

            $copy1 = copy($mails_dir.'backup.txt', $mails_dir.$iso.'/backup.txt');
            $copy2 = copy($mails_dir.'backup.html', $mails_dir.$iso.'/backup.html');
            if (!$copy1 or !$copy2) {
                $this->html .= $this->displayError(
                    $this->l('Cannot create email template. Please check for write access.')
                );

                return false;
            } else {
                return true;
            }
        }
    }

    /**
     * @param int|null $id_shop
     * @param bool|null $ssl
     * @param bool|false $relative_protocol
     * @return string
     */
    protected function getBaseLink($id_shop = null, $ssl = null, $relative_protocol = false)
    {
        static $force_ssl = null;
        $ssl_enable = Configuration::get('PS_SSL_ENABLED');

        if ($ssl === null) {
            if ($force_ssl === null) {
                $force_ssl = (Configuration::get('PS_SSL_ENABLED') && Configuration::get('PS_SSL_ENABLED_EVERYWHERE'));
            }
            $ssl = $force_ssl;
        }

        $context = Context::getContext();

        if (Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') && $id_shop !== null) {
            $shop = new Shop($id_shop);
        } else {
            $shop = $context->shop;
        }

        if ($relative_protocol) {
            $base = '//'.($ssl && $ssl_enable ? $shop->domain_ssl : $shop->domain);
        } else {
            $base = (($ssl && $ssl_enable) ? 'https://'.$shop->domain_ssl : 'http://'.$shop->domain);
        }

        return $base.$shop->getBaseURI();
    }

    public function getModulePathUrl()
    {
        return $this->getBaseLink().'modules/'.$this->name.'/';
    }

    /**
     * @param string $type , 'db' or 'files'
     */
    protected function deleteOldBackups($type)
    {
        $backup_dirs = $this->getBackupDirs();
        //search all files in the db backup dir:
        $files = scandir($backup_dirs[$type]);

        $limit = ($type == 'db' ? $this->max_db_backups : $this->max_files_backups);
        if (!$limit) {
            return;
        }

        //filter real backups:
        $files_result = array();
        foreach ($files as $file) {
            if (strpos($file, $this->backup_suffix) === false) {
                continue;
            }
            $new_key = explode('-', $file);
            //if filename have first part and it's a time:
            if (isset($new_key[0])) {
                $new_key = $new_key[0];
                $files_result[$new_key] = $file;
            }
        }

        //detect the latest backups
        krsort($files_result);
        $latest_keys = array_slice($files_result, 0, $limit);

        //delete old backups
        foreach ($files_result as $key => $file) {
            if (!in_array($key, $latest_keys)) {
                @unlink($backup_dirs[$type].$file);
            }
        }
    }

    /**
     * Create .zip archive
     * @param string $source
     * @param string $destination
     * @return bool|string, false if failed or .zip path if success
     */
    protected function zip($source, $destination)
    {
        if (!file_exists($source)) {
            return false;
        }
        if (class_exists('ZipArchive')) {
            $destination .= '.zip';

            try {
                $zip = new ZipArchive();
                if (!$zip->open($destination, ZIPARCHIVE::CREATE)) {
                    return false;
                }

                $source = str_replace('\\', '/', realpath($source));
                $source = rtrim($source, '/').'/';

                if (is_dir($source) === true) {
                    $files = new RecursiveIteratorIterator(
                        new RecursiveDirectoryIterator($source, FilesystemIterator::SKIP_DOTS),
                        RecursiveIteratorIterator::SELF_FIRST
                    );

                    foreach ($files as $file) {
                        $file = $file->getRealPath();
                        $file = str_replace('\\', '/', $file);

                        if (!$this->checkFileForAdding($file)) {
                            continue;
                        }

                        $new_name = str_replace(dirname($source), '', $file);
                        $new_name = ltrim($new_name, '/');
                        if (is_dir($file) === true) {
                            $zip->addEmptyDir($new_name.'/');
                        } elseif (is_file($file) === true) {
                            $zip->addFile($file, $new_name);
                        }
                    }
                } else {
                    if (is_file($source) === true) {
                        $zip->addFromString($source, basename($source));
                    }
                }

                $zip->close();

                return $destination;
            } catch (Exception $e) {
                $this->html .= '<pre>'.$this->displayError("Exception : ".$e).'</pre>';
            }
        // else if zip class is not available
        } else {
            require_once(_PS_ROOT_DIR_.'/tools/pclzip/pclzip.lib.php');
            $destination .= '.zip';
            $zip = new PclZip($destination);
            $v_list = $zip->add($source, '', dirname($source));
            if ($v_list == 0) {
                $this->html .= '<pre>'.$this->displayError("Error : ".$zip->errorInfo(true)).'</pre>';
            }

            if (file_exists($destination)) {
                // remove unnecessary files
                $zip->delete(PCLZIP_OPT_BY_PREG, '/.*'.preg_quote($this->backup_suffix).'.*|.*pclzip.*\.tmp.*/');
            }

            return $destination;
        }

        return false;
    }

    /**
     * Create tar.gz using exec
     * @param string $source
     * @param string $destination
     * @return bool|string, false if failed or .tar.gz path if success
     */
    protected function targzExec($source, $destination)
    {
        if (!file_exists($source)) {
            return false;
        }

        $destination .= '.tar.gz';

        try {
            //todo check for exec result
            chdir(dirname($source));
            $command = 'tar -czf '.escapeshellarg($destination).' '.escapeshellarg(basename($source)).' --exclude='.
                escapeshellarg('*'.$this->backup_suffix.'*');
            // Exclude cache
            if ($this->files_ignore_cache) {
                foreach ($this->exclude_paths_cache as $path) {
                    $command .= ' --exclude='.escapeshellarg('*'.$path.'*');
                }
            }

            // Exclude dirs
            foreach ($this->getExcludePaths() as $path) {
                $command .= ' --exclude='.escapeshellarg('*'.$path.'*');
            }

            exec($command);

            return $destination;
        } catch (Exception $e) {
            $this->html .= '<pre>'.$this->displayError("Exception : ".$e).'</pre>';
        }

        return false;
    }

    /**
     * Create tar.gz
     * @param string $source
     * @param string $destination
     * @return bool|string, false if failed or .tar.gz path if success
     */
    protected function targz($source, $destination)
    {
        if (!file_exists($source)) {
            return false;
        }

        // todo: PharData skips empty folders, it should be fixed for normal usage
        try {
            $destination .= '.tar';
            //build archive
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($source, FilesystemIterator::SKIP_DOTS)
            );

            $filterIterator = new CallbackFilterIterator($iterator, array($this, 'checkFileForAdding'));

            $pd = new PharData($destination);
            $pd->buildFromIterator($filterIterator, $source);
            $pd->compress(Phar::GZ);

            @unlink($destination);

            return $destination.'.gz';
        } catch (Exception $e) {
            $this->html .= '<pre>'.$this->displayError("Exception : ".$e).'</pre>';
        }

        return false;
    }

    /**
     * @return string, generated html
     */
    protected function renderWarning()
    {
        $this->context->smarty->assign(array(
            'psv' => $this->getPSVersion(),
            'hide_warning' => Configuration::get($this->settings_prefix.'HIDE_WARNING'),
        ));

        return $this->context->smarty->fetch($this->local_path . 'views/templates/admin/warning.tpl');
    }

    protected function renderBackupsTable()
    {
        $types = array('db' => $this->l('Database'), 'files' => $this->l('Files'));

        $avail_backups = array();
        foreach ($types as $type => $type_name) {
            $avail_backups[$type] = $this->getBackups($type);
        }

        $this->context->smarty->assign(array(
            'psv' => $this->getPSVersion(),
            'types' => $types,
            'avail_backups' => $avail_backups,
        ));

        return $this->context->smarty->fetch($this->local_path . 'views/templates/admin/backups_table.tpl');
    }

    public function getBackups($type)
    {
        // Local backups:
        $backup_dirs = $this->getBackupDirs();
        //search all files in the db backup dir:
        $files = scandir($backup_dirs[$type]);
        $results = array();

        foreach ($files as $file) {
            if (strpos($file, $this->backup_suffix) === false) {
                continue;
            }

            $result = array();
            $result['filename'] = $file;

            // get creation time:
            $parts = explode('-', $file);
            //if filename have first part and it's a time:
            if (isset($parts[0])) {
                $result['created'] = date('Y-m-d H:i:s', $parts[0]);
                $result['created_time'] = $parts[0];
            } else {
                $result['created'] = '';
            }

            $result['url'] = $this->getBaseLink().'modules/'.$this->name.
                '/url.php?filename='.basename($file).'&type='.$type;

            $filesize = filesize($backup_dirs[$type].$file);
            $filesize = $filesize ? ($filesize / 1024 / 1024) : 0;
            $result['filesize'] = $filesize;

            $results[$file] = $result;
        }
        
        foreach ($results as &$item) {
            $item['comment'] = $this->getComment($item['filename']);
        }

        if (sizeof($results)) {
            usort($results, array($this, 'sortByTime'));
            $results = array_reverse($results);
        }

        return $results;
    }

    /**
     * Check if file should be added to backup or ignored
     * @param $file_path string
     * @return bool
     */
    public function checkFileForAdding($file_path)
    {
        $file_path = rtrim($file_path, '/');

        // ignore existing backups
        if (strpos($file_path, $this->backup_suffix) !== false) {
            return false;
        }

        //ignore unnecessary files
        if (basename($file_path) == '.directory') {
            return false;
        }

        // don't ignore any index.php
        if (strpos($file_path, 'index.php') !== false) {
            return true;
        }

        // ignore cache
        if ($this->files_ignore_cache) {
            foreach ($this->exclude_paths_cache as $path) {
                if (strpos($file_path, $path) !== false) {
                    return false;
                }
            }
        }

        // Exclude dirs
        foreach ($this->getExcludePaths() as $path) {
            if (strpos($file_path, $path) !== false) {
                return false;
            }
        }

        return true;
    }

    public function sortByTime($a, $b)
    {
        return $a['created_time'] - $b['created_time'];
    }

    public function restoreBackup($file_path, $type)
    {
        $result = false;

        $shop_enable = Configuration::get('PS_SHOP_ENABLE');
        // turn on maintenance
        Configuration::updateValue('PS_SHOP_ENABLE', 0);

        if (file_exists($file_path)) {
            if ($type == 'db') {
                $result = $this->restoreDb($file_path);
            }
        }

        // set maintenance to previous value
        Configuration::updateValue('PS_SHOP_ENABLE', $shop_enable);

        return $result;
    }

    protected function restoreDb($file_path)
    {
        $result = false;

        $extension = pathinfo($file_path, PATHINFO_EXTENSION);

        switch ($extension) {
            case 'bz':
            case 'bz2':
                $fp = bzopen($file_path, 'r');
                break;
            case 'gz':
                $fp = gzopen($file_path, 'r');
                break;
            default:
                $fp = fopen($file_path, 'r');
        }

        if (is_resource($fp) === true) {
            $query = array();

            while (feof($fp) === false) {
                $query[] = fgets($fp);

                if (preg_match('~'.preg_quote(';', '~').'\s*$~iS', end($query)) === 1) {
                    $query = trim(implode('', $query));

                    // check if at least one query was successful
                    if (Db::getInstance()->execute($query)) {
                        $result = true;
                    }

                    while (ob_get_level() > 0) {
                        ob_end_flush();
                    }

                    flush();
                }

                if (is_string($query) === true) {
                    $query = array();
                }
            }

            switch ($extension) {
                case 'bz':
                case 'bz2':
                    bzclose($fp);
                    break;
                case 'gz':
                    gzclose($fp);
                    break;
                default:
                    fclose($fp);
            }
        } else {
            return false;
        }

        return $result;
    }

    public function renderAdditionalData()
    {
        $this->context->smarty->assign(array());

        return $this->context->smarty->fetch($this->local_path . 'views/templates/admin/additional.tpl');
    }

    protected function findRootDirInBackup($path)
    {
        if (!is_dir($path)) {
            return false;
        }
        $path = rtrim($path, '/').'/';

        $needles = array('classes', 'config', 'controllers', 'img', 'override', 'themes');
        $elems = array_diff(scandir($path), array('..', '.'));
        if (!array_diff($needles, $elems)) {
            return $path;
        } else {
            foreach ($elems as $elem) {
                $elem_path = $path.$elem;
                if (is_dir($elem_path)) {
                    $result = $this->findRootDirInBackup($elem_path);
                    if ($result) {
                        return $result;
                    }
                }
            }
        }

        return false;
    }

    protected function removeDir($path, $removeRoot = true, $exclude = null)
    {
        if (is_dir($path) === true) {
            $files = array_diff(scandir($path), array('.', '..'));

            foreach ($files as $file) {
                $file_path = realpath($path).'/'.$file;
                $file_path = str_replace('\\', '/', $file_path);
                if (!$exclude or ($exclude and strpos($file_path, $exclude) === false)) {
                    $this->removeDir($file_path, true, $exclude);
                }
            }

            if ($removeRoot) {
                return @rmdir($path);
            } else {
                return true;
            }
        } else {
            if (is_file($path) === true) {
                return @unlink($path);
            }
        }

        return false;
    }

    protected function recurseCopy($source, $dest)
    {
        $source = rtrim($source, '/');
        $dest = rtrim($dest, '/');

        if (!is_dir($dest)) {
            mkdir($dest, 0755, true);
        }

        $dir = opendir($source);
        while (false !== ($file = readdir($dir))) {
            if (($file != '.') && ($file != '..')) {
                if (is_dir($source.'/'.$file)) {
                    $this->recurseCopy($source.'/'.$file, $dest.'/'.$file);
                } else {
                    $copy_result = @copy($source.'/'.$file, $dest.'/'.$file);
                    if (!$copy_result) {
                        $dest_path = str_replace(_PS_ROOT_DIR_, '', $dest.'/'.$file);
                        $this->restore_errors[] =
                            sprintf($this->l('Unable to overwrite file %s'), '<b>'.$dest_path.'</b>');
                    }
                }
            }
        }
        closedir($dir);
    }

    protected function checkMaxExecutionTime()
    {
        $html = '';

        $max_execution_time = ini_get('max_execution_time');

        if ($max_execution_time == 0 or $max_execution_time >= 14400) {
            return '';
        } elseif ($max_execution_time >= 60) {
            $html .= $this->displayWarning(
                $this->l('Max execution time is restricted.
                It can cause errors while making or restoring backups.
                Contact your hosting provider in order to increase PHP value max_execution_time
                or to allow the module to change it.')
            );
        } else {
            $html .= $this->displayError(
                $this->l('Max execution time is restricted.
                It can cause errors while making or restoring backups.
                Contact your hosting provider in order to increase PHP value max_execution_time
                or to allow the module to change it.')
            );
        }

        return $html;
    }

    /**
     * Check if we can use exec and tar
     * @return bool
     */
    protected function checkExecAvail()
    {
        // check if exec disabled:
        $disabled = explode(',', ini_get('disable_functions'));
        if (in_array('exec', $disabled)) {
            return false;
        }

        // check if exec tar is working:
        $source_file = _PS_MODULE_DIR_.$this->name.'/'.$this->name.'.php';
        $result_file = _PS_MODULE_DIR_.$this->name.'/backup/files/'.$this->name.'.php.tar.gz';
        @unlink($result_file); // just in case

        $cmd = 'tar -czf '.escapeshellarg($result_file).' '.escapeshellarg($source_file);
        @exec($cmd);

        if (file_exists($result_file)) {
            @unlink($result_file);
            return true;
        } else {
            return false;
        }
    }

    protected function getExcludePaths($with_cache = false)
    {
        $exclude_paths_custom = str_replace("\n", ';', str_replace("\r", '', $this->exclude_paths_custom));
        $exclude_paths_custom = trim($exclude_paths_custom);
        if ($exclude_paths_custom) {
            $exclude_paths_custom = explode(';', $exclude_paths_custom);
        }
        if (is_array($exclude_paths_custom)) {
	        foreach ($exclude_paths_custom as &$path) {
	            $path = ltrim($path, '/');
	            $path = _PS_ROOT_DIR_.'/'.$path;
	        }
	        $exclude_paths_custom = array_merge($exclude_paths_custom, $this->exclude_paths_always);
	        if ($with_cache) {
	            return array_merge($this->exclude_paths_cache, $exclude_paths_custom);
	        } else {
	            return $exclude_paths_custom;
	        }
	    }

		if ($with_cache) {
			return array_merge($this->exclude_paths_cache, $this->exclude_paths_always);
		}

	    return $this->exclude_paths_always;
    }

    protected function checkCreateTableComments()
    {
        Db::getInstance()->execute('
		CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'psbp_comments` (
			`id_comment` INT(6) NOT NULL AUTO_INCREMENT,
			`filename` VARCHAR(255) NOT NULL,
			`comment` VARCHAR(255) NOT NULL,
			PRIMARY KEY(`id_comment`), UNIQUE(`filename`, `comment`)
		) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8');
    }

    protected function addComment($filename, $comment)
    {
        Db::getInstance()->execute(
            'INSERT IGNORE INTO `' . _DB_PREFIX_ . 'psbp_comments`
            (`filename`, `comment`)
            VALUES ("' . pSQL($filename) . '", "' . pSQL($comment) . '")'
        );
    }

    protected function getComment($filename)
    {
        $comment = Db::getInstance()->getValue(
            'SELECT `comment`
            FROM `' . _DB_PREFIX_ . 'psbp_comments`
            WHERE `filename` = "'.pSQL($filename).'"'
        );

        return $comment;
    }

    protected function getConfirmationMsg($code)
    {
        switch ($code) {
            case 1:
                return $this->l('Settings updated.');
                break;
            case 2:
                return $this->l('Db backup successful.');
                break;
            case 3:
                return $this->l('Db backup saved to Google Drive successfully.');
                break;
            case 4:
                return $this->l('Files backup successful.');
                break;
            case 5:
                return $this->l('Files backup saved to Google Drive successfully.');
                break;
        }
    }
}
