<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * @author    Presta.Site
 * @copyright 2015 Presta.Site
 * @license   LICENSE.txt
 */

@ini_set('max_execution_time', 14400);

class PSDbBackup
{
    /** @var integer Object id */
    public $id;
    /** @var string Last error messages */
    public $error;
    /** @var string default backup directory. */
    public static $backupDir = '/modules/prestabackup/backup/db/';
    /** @var string custom backup directory. */
    public $customBackupDir = null;

    public $psBackupAll = true;
    public $psBackupDropTable = true;
    public $backup_suffix = '-psbckp';

    /**
     * Creates a new backup object
     *
     * @param string $filename Filename of the backup file
     */
    public function __construct($filename = null)
    {
        if ($filename) {
            $this->id = $this->getRealBackupPath($filename);
        }

        $psBackupAll = Configuration::get('PSB_BACKUP_ALL');
        $psBackupDropTable = Configuration::get('PSB_DROP_TABLE');
        $this->psBackupAll = $psBackupAll !== false ? $psBackupAll : true;
        $this->psBackupDropTable = $psBackupDropTable !== false ? $psBackupDropTable : true;
    }

    /**
     * you can set a different path with that function
     *
     * @TODO include the prefix name
     * @param string $dir
     * @return boolean bo
     */
    public function setCustomBackupPath($dir)
    {
        $customDir = DIRECTORY_SEPARATOR.trim($dir, '/').DIRECTORY_SEPARATOR;
        if (is_dir(_PS_ROOT_DIR_.DIRECTORY_SEPARATOR.$customDir.DIRECTORY_SEPARATOR)) {
            $this->customBackupDir = $customDir;
        } else {
            return false;
        }

        return true;

    }

    /**
     * get the path to use for backup (customBackupDir if specified, or default)
     *
     * @param string $filename filename to use
     * @return string full path
     */
    public function getRealBackupPath($filename = null)
    {
        $backupDir = PSDbBackup::getBackupPath($filename);
        if (!empty($this->customBackupDir)) {
            $backupDir = str_replace(
                _PS_ROOT_DIR_.self::$backupDir,
                _PS_ROOT_DIR_.$this->customBackupDir,
                $backupDir
            );

            if (strrpos($backupDir, DIRECTORY_SEPARATOR)) {
                $backupDir .= DIRECTORY_SEPARATOR;
            }
        }

        return $backupDir;
    }

    /**
     * Get the full path of the backup file
     *
     * @param string $filename prefix of the backup file (datetime will be the second part)
     * @param string $backupdir custom backup dir
     * @return The full path of the backup file, or false if the backup file does not exists
     */
    public static function getBackupPath($filename = '', $backupdir = null)
    {
        $backupdir = $backupdir ? realpath(_PS_ROOT_DIR_.$backupdir) : realpath(_PS_ROOT_DIR_.self::$backupDir);

        if ($backupdir === false) {
            die(Tools::displayError('"Backup" directory does not exist.'));
        }

        // Check the realpath so we can validate the backup file is under the backup directory
        if (!empty($filename)) {
            $backupfile = realpath($backupdir.DIRECTORY_SEPARATOR.$filename);
        } else {
            $backupfile = $backupdir.DIRECTORY_SEPARATOR;
        }

        if ($backupfile === false || strncmp($backupdir, $backupfile, Tools::strlen($backupdir)) != 0) {
            die(Tools::displayError());
        }

        return $backupfile;
    }

    /**
     * Check if a backup file exist
     *
     * @param string $filename prefix of the backup file (datetime will be the second part)
     * @return boolean true if backup file exist
     */
    public static function backupExist($filename)
    {
        $backupdir = realpath(_PS_ROOT_DIR_.self::$backupDir);

        if ($backupdir === false) {
            die(Tools::displayError('"Backup" directory does not exist.'));
        }

        return @filemtime($backupdir.DIRECTORY_SEPARATOR.$filename);
    }

    /**
     * Get the URL used to retreive this backup file
     *
     * @return The url used to request the backup file
     */
    public function getBackupURL()
    {
        return $this->getBaseLink().'modules/prestabackup/url.php?filename='.basename($this->id);
    }

    /**
     * Delete the current backup file
     *
     * @return boolean Deletion result, true on success
     */
    public function delete()
    {
        if (!$this->id || !unlink($this->id)) {
            $this->error = Tools::displayError('Error deleting').' '.($this->id ? '"'.$this->id.'"' :
                    Tools::displayError('Invalid ID'));

            return false;
        }

        return true;
    }

    /**
     * Deletes a range of backup files
     *
     * @return boolean True on success
     */
    public function deleteSelection($list)
    {
        foreach ($list as $file) {
            $backup = new PSDbBackup($file);
            if (!$backup->delete()) {
                $this->error = $backup->error;

                return false;
            }
        }

        return true;
    }

    /**
     * Creates a new backup file
     *
     * @return boolean true on successful backup
     */
    public function add()
    {
        if (!$this->psBackupAll) {
            $ignore_insert_table = array(
                _DB_PREFIX_.'connections',
                _DB_PREFIX_.'connections_page',
                _DB_PREFIX_
                .'connections_source',
                _DB_PREFIX_.'guest',
                _DB_PREFIX_.'statssearch',
            );
        } else {
            $ignore_insert_table = array();
        }

        // Generate some random number, to make it extra hard to guess backup file names
        $rand = dechex(mt_rand(0, min(0xffffffff, mt_getrandmax())));
        $date = time();
        $backupfile = $this->getRealBackupPath().$date.'-db-'.$rand.$this->backup_suffix.'.sql';

        // Figure out what compression is available and open the file
        if (function_exists('bzopen')) {
            $backupfile .= '.bz2';
            $fp = @bzopen($backupfile, 'w');
        } elseif (function_exists('gzopen')) {
            $backupfile .= '.gz';
            $fp = @gzopen($backupfile, 'w');
        } else {
            $fp = @fopen($backupfile, 'w');
        }

        if ($fp === false) {
            echo Tools::displayError('Unable to create backup file').' "'.addslashes($backupfile).'"';

            return false;
        }

        $this->id = realpath($backupfile);

        fwrite(
            $fp,
            '/* Backup for '.Tools::getHttpHost(false, false).__PS_BASE_URI__."\n *  at ".date($date)."\n */\n"
        );
        fwrite($fp, "\n".'SET NAMES \'utf8\';'."\n\n");

        // Find all tables
        $tables = Db::getInstance()->executeS('SHOW TABLES');
        $found = 0;
        foreach ($tables as $table) {
            $table = current($table);

            // Skip tables which do not start with _DB_PREFIX_
            if (Tools::strlen($table) < Tools::strlen(_DB_PREFIX_) ||
                strncmp($table, _DB_PREFIX_, Tools::strlen(_DB_PREFIX_)) != 0) {
                continue;
            }

            // Export the table schema
            $schema = Db::getInstance()->executeS('SHOW CREATE TABLE `'.$table.'`');

            if (count($schema) != 1 || !isset($schema[0]['Table']) || !isset($schema[0]['Create Table'])) {
                fclose($fp);
                $this->delete();
                echo Tools::displayError('An error occurred while backing up. Unable to obtain the schema of').' "'.
                    $table;

                return false;
            }

            fwrite($fp, '/* Scheme for table '.$schema[0]['Table']." */\n");

            if ($this->psBackupDropTable) {
                fwrite($fp, 'DROP TABLE IF EXISTS `'.$schema[0]['Table'].'`;'."\n");
            }

            fwrite($fp, $schema[0]['Create Table'].";\n\n");

            if (!in_array($schema[0]['Table'], $ignore_insert_table)) {
                $data = Db::getInstance()->query('SELECT * FROM `'.$schema[0]['Table'].'`', false);
                $sizeof = DB::getInstance()->NumRows();
                $lines = explode("\n", $schema[0]['Create Table']);

                if ($data && $sizeof > 0) {
                    // Export the table data
                    fwrite($fp, 'INSERT INTO `'.$schema[0]['Table']."` VALUES\n");
                    $i = 1;
                    while ($row = DB::getInstance()->nextRow($data)) {
                        $s = '(';

                        foreach ($row as $field => $value) {
                            $tmp = "'".pSQL($value, true)."',";
                            if ($tmp != "'',") {
                                $s .= $tmp;
                            } else {
                                foreach ($lines as $line) {
                                    if (strpos($line, '`'.$field.'`') !== false) {
                                        if (preg_match('/(.*NOT NULL.*)/Ui', $line)) {
                                            $s .= "'',";
                                        } else {
                                            $s .= 'NULL,';
                                        }
                                        break;
                                    }
                                }
                            }
                        }
                        $s = rtrim($s, ',');

                        if ($i % 200 == 0 && $i < $sizeof) {
                            $s .= ");\nINSERT INTO `".$schema[0]['Table']."` VALUES\n";
                        } elseif ($i < $sizeof) {
                            $s .= "),\n";
                        } else {
                            $s .= ");\n";
                        }

                        fwrite($fp, $s);
                        ++$i;
                    }
                }
            }
            $found++;
        }

        fclose($fp);
        if ($found == 0) {
            $this->delete();
            echo Tools::displayError('No valid tables were found to backup.');

            return false;
        }

        return true;
    }

    protected function getBaseLink($id_shop = null, $ssl = null, $relative_protocol = false)
    {
        static $force_ssl = null;
        $ssl_enable = Configuration::get('PS_SSL_ENABLED');

        if ($ssl === null) {
            if ($force_ssl === null) {
                $force_ssl = (Configuration::get('PS_SSL_ENABLED') && Configuration::get('PS_SSL_ENABLED_EVERYWHERE'));
            }
            $ssl = $force_ssl;
        }

        $context = Context::getContext();

        if (Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') && $id_shop !== null) {
            $shop = new Shop($id_shop);
        } else {
            $shop = $context->shop;
        }

        if ($relative_protocol) {
            $base = '//'.($ssl && $ssl_enable ? $shop->domain_ssl : $shop->domain);
        } else {
            $base = (($ssl && $ssl_enable) ? 'https://'.$shop->domain_ssl : 'http://'.$shop->domain);
        }

        return $base.$shop->getBaseURI();
    }
}
