<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author     SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZend_Controller
 * @category   SZend
 */

/** SZendControllerRequestException */
require_once dirname(__FILE__).'/../../Controller/Request/Exception.php';

/** SZendControllerRequestAbstract */
require_once dirname(__FILE__).'/../../Controller/Request/Abstract.php';

/** SZendUri */
require_once dirname(__FILE__).'/../../Uri.php';

/**
 * SZendControllerRequestHttp
 *
 * HTTP request object for use with SZend_Controller family.
 *
 * @uses SZendControllerRequestAbstract
 * @package SZend_Controller
 * @subpackage Request
 */
class SZendControllerRequestHttp extends SZendControllerRequestAbstract
{
    /**
     * Allowed parameter sources
     * @var array
     */
    protected $paramSources = array('_GET', '_POST');

    /**
     * REQUEST_URI
     * @var string;
     */
    protected $requestUri;

    /**
     * Base URL of request
     * @var string
     */
    protected $baseUrl = null;

    /**
     * Base path of request
     * @var string
     */
    protected $basePath = null;

    /**
     * PATH_INFO
     * @var string
     */
    protected $pathInfo = '';

    /**
     * Instance parameters
     * @var array
     */
    protected $params = array();

    /**
     * Alias keys for request parameters
     * @var array
     */
    protected $aliases = array();

    /**
     * Constructor
     *
     * If a $uri is passed, the object will attempt to populate itself using
     * that information.
     *
     * @param string|SZendUri $uri
     * @return void
     * @throws SZendControllerRequestException when invalid URI passed
     */
    public function __construct($uri = null)
    {
        if (null !== $uri) {
            if (!$uri instanceof SZendUri) {
                $uri = SZendUri::factory($uri);
            }
            if ($uri->valid()) {
                $path = $uri->getPath();
                $query = $uri->getQuery();
                if (!empty($query)) {
                    $path .= '?'.$query;
                }

                $this->setRequestUri($path);
            } else {
                require_once 'SZend/Controller/Request/Exception.php';
                throw new SZendControllerRequestException('Invalid URI provided to constructor');
            }
        } else {
            $this->setRequestUri();
        }
    }

    /**
     * Alias to __get
     *
     * @param string $key
     * @return mixed
     */
    public function get($key)
    {
        return $this->__get($key);
    }

    /**
     * Access values contained in the superglobals as public members
     * Order of precedence: 1. GET, 2. POST, 3. COOKIE, 4. SERVER, 5. ENV
     *
     * @see http://msdn.microsoft.com/en-us/library/system.web.httprequest.item.aspx
     * @param string $key
     * @return mixed
     */
    public function __get($key)
    {

        switch (true) {
            case array_key_exists($key, $this->params):
                return $this->params[$key];
            case Tools::getIsset($key):
                return Tools::getValue($key);
            case array_key_exists($key, Context::getContext()->cookie) && Context::getContext()->cookie[$key]:
                return Context::getContext()->cookie[$key];
            case ($key == 'REQUEST_URI'):
                return $this->getRequestUri();
            case ($key == 'PATH_INFO'):
                return $this->getPathInfo();
            case array_key_exists($key, $_SERVER):
                return $_SERVER[$key];
            case array_key_exists($key, $_ENV):
                return $_ENV[$key];
            default:
                return null;
        }
    }

    /**
     * Set values
     *
     * In order to follow {@link __get()}, which operates on a number of
     * superglobals, setting values through overloading is not allowed and will
     * raise an exception. Use setParam() instead.
     *
     * @param string $key
     * @param mixed $value
     * @return void
     * @throws SZendControllerRequestException
     */
    public function __set($key, $value)
    {
        require_once 'SZend/Controller/Request/Exception.php';
        if (isset($key) || isset($value) || true) {
            throw new SZendControllerRequestException(
                'Setting values in superglobals not allowed; please use setParam()'
            );
        }
    }

    /**
     * Returns the REQUEST_URI taking into account
     * platform differences between Apache and IIS
     *
     * @return string
     */
    public function getRequestUri()
    {
        if (empty($this->requestUri)) {
            $this->setRequestUri();
        }

        return $this->requestUri;
    }

    /**
     * Set the REQUEST_URI on which the instance operates
     *
     * If no request URI is passed, uses the value in $_SERVER['REQUEST_URI'],
     * $_SERVER['HTTP_X_REWRITE_URL'], or $_SERVER['ORIG_PATH_INFO'] + $_SERVER['QUERY_STRING'].
     *
     * @param string $requestUri
     * @return SZendControllerRequestHttp
     */
    public function setRequestUri($requestUri = null)
    {
        if ($requestUri === null) {
            if (array_key_exists('HTTP_X_REWRITE_URL', $_SERVER) &&
                !empty($_SERVER['HTTP_X_REWRITE_URL'])) { // check this first so IIS will catch
                $requestUri = $_SERVER['HTTP_X_REWRITE_URL'];
            } elseif (array_key_exists('REQUEST_URI', $_SERVER) &&
                !empty($_SERVER['REQUEST_URI'])) {
                $requestUri = $_SERVER['REQUEST_URI'];
            } elseif (array_key_exists('ORIG_PATH_INFO', $_SERVER) &&
                !empty($_SERVER['ORIG_PATH_INFO'])) { // IIS 5.0, PHP as CGI
                $requestUri = $_SERVER['ORIG_PATH_INFO'];
                if (array_key_exists('QUERY_STRING', $_SERVER) &&
                    !empty($_SERVER['QUERY_STRING'])) {
                    $requestUri .= '?'.$_SERVER['QUERY_STRING'];
                }
            } else {
                return $this;
            }
        } elseif (!is_string($requestUri)) {
            return $this;
        } else {
            // Set GET items, if available
            $_GET = array();
            if (false !== ($pos = strpos($requestUri, '?'))) {
                // Get key => value pairs and set $_GET
                $query = Tools::substr($requestUri, $pos + 1);
                parse_str($query, $vars);
                $_GET = $vars;
            }
        }

        $this->requestUri = $requestUri;

        return $this;
    }

    /**
     * Returns everything between the BaseUrl and QueryString.
     * This value is calculated instead of reading PATH_INFO
     * directly from $_SERVER due to cross-platform differences.
     *
     * @return string
     */
    public function getPathInfo()
    {
        if (empty($this->pathInfo)) {
            $this->setPathInfo();
        }

        return $this->pathInfo;
    }

    /**
     * Set the PATH_INFO string
     *
     * @param string|null $pathInfo
     * @return SZendControllerRequestHttp
     */
    public function setPathInfo($pathInfo = null)
    {
        if ($pathInfo === null) {
            $baseUrl = $this->getBaseUrl();

            if (null === ($requestUri = $this->getRequestUri())) {
                return $this;
            }

            // Remove the query string from REQUEST_URI
            if ($pos = strpos($requestUri, '?')) {
                $requestUri = Tools::substr($requestUri, 0, $pos);
            }

            if ((null !== $baseUrl)
                && (false === ($pathInfo = Tools::substr($requestUri, Tools::strlen($baseUrl))))) {
                // If Tools::substr() returns false then PATH_INFO is set to an empty string
                $pathInfo = '';
            } elseif (null === $baseUrl) {
                $pathInfo = $requestUri;
            }
        }

        $this->pathInfo = (string)$pathInfo;

        return $this;
    }

    /**
     * Everything in REQUEST_URI before PATH_INFO
     * <form action="<?=$baseUrl?>/news/submit" method="POST"/>
     *
     * @return string
     */
    public function getBaseUrl()
    {
        if (null === $this->baseUrl) {
            $this->setBaseUrl();
        }

        return $this->baseUrl;
    }

    /**
     * Set the base URL of the request; i.e., the segment leading to the script name
     *
     * E.g.:
     * - /admin
     * - /myapp
     * - /subdir/index.php
     *
     * Do not use the full URI when providing the base. The following are
     * examples of what not to use:
     * - http://example.com/admin (should be just /admin)
     * - http://example.com/subdir/index.php (should be just /subdir/index.php)
     *
     * If no $baseUrl is provided, attempts to determine the base URL from the
     * environment, using SCRIPT_FILENAME, SCRIPT_NAME, PHP_SELF, and
     * ORIG_SCRIPT_NAME in its determination.
     *
     * @param mixed $baseUrl
     * @return SZendControllerRequestHttp
     */
    public function setBaseUrl($baseUrl = null)
    {
        if ((null !== $baseUrl) && !is_string($baseUrl)) {
            return $this;
        }

        if ($baseUrl === null) {
            $filename = basename($_SERVER['SCRIPT_FILENAME']);

            if (basename($_SERVER['SCRIPT_NAME']) === $filename) {
                $baseUrl = $_SERVER['SCRIPT_NAME'];
            } elseif (basename($_SERVER['PHP_SELF']) === $filename) {
                $baseUrl = $_SERVER['PHP_SELF'];
            } elseif (array_key_exists('ORIG_SCRIPT_NAME', $_SERVER) &&
                basename($_SERVER['ORIG_SCRIPT_NAME']) === $filename) {
                $baseUrl = $_SERVER['ORIG_SCRIPT_NAME']; // 1and1 shared hosting compatibility
            } else {
                // Backtrack up the script_filename to find the portion matching
                // php_self
                $path = $_SERVER['PHP_SELF'];
                $segs = explode('/', trim($_SERVER['SCRIPT_FILENAME'], '/'));
                $segs = array_reverse($segs);
                $index = 0;
                $last = count($segs);
                $baseUrl = '';
                do {
                    $seg = $segs[$index];
                    $baseUrl = '/'.$seg.$baseUrl;
                    ++$index;
                } while (($last > $index) && (false !== ($pos = strpos($path, $baseUrl))) && (0 != $pos));
            }

            // Does the baseUrl have anything in common with the request_uri?
            $requestUri = $this->getRequestUri();

            if (0 === strpos($requestUri, $baseUrl)) {
                // full $baseUrl matches
                $this->baseUrl = $baseUrl;

                return $this;
            }

            if (0 === strpos($requestUri, dirname($baseUrl))) {
                // directory portion of $baseUrl matches
                $this->baseUrl = rtrim(dirname($baseUrl), '/');

                return $this;
            }

            if (!strpos($requestUri, basename($baseUrl))) {
                // no match whatsoever; set it blank
                $this->baseUrl = '';

                return $this;
            }

            // If using mod_rewrite or ISAPI_Rewrite strip the script filename
            // out of baseUrl. $pos !== 0 makes sure it is not matching a value
            // from PATH_INFO or QUERY_STRING
            if ((Tools::strlen($requestUri) >= Tools::strlen($baseUrl))
                && ((false !== ($pos = strpos($requestUri, $baseUrl))) && ($pos !== 0))) {
                $baseUrl = Tools::substr($requestUri, 0, $pos + Tools::strlen($baseUrl));
            }
        }

        $this->baseUrl = rtrim($baseUrl, '/');

        return $this;
    }

    /**
     * Alias to __set()
     *
     * @param string $key
     * @param mixed $value
     * @return void
     */
    public function set($key, $value)
    {
        return $this->__set($key, $value);
    }

    /**
     * Alias to __isset()
     *
     * @param string $key
     * @return boolean
     */
    public function has($key)
    {
        return $this->__isset($key);
    }

    /**
     * Check to see if a property is set
     *
     * @param string $key
     * @return boolean
     */
    public function __isset($key)
    {
        switch (true) {
            case array_key_exists($key, $this->params) && $this->params[$key]:
            case Tools::getIsset($key):
            case array_key_exists($key, Context::getContext()->cookie) && Context::getContext()->cookie[$key]:
            case array_key_exists($key, $_SERVER) && $_SERVER[$key]:
            case array_key_exists($key, $_ENV) && $_ENV[$key]:
                return true;
            default:
                return false;
        }
    }

    /**
     * Retrieve a member of the $_GET superglobal
     *
     * If no $key is passed, returns the entire $_GET array.
     *
     * @param string $key
     * @param mixed $default Default value to use if key not found
     * @return mixed Returns null if key does not exist
     * @todo How to retrieve from nested arrays
     */
    public function getQuery($key = null, $default = null)
    {
        if (null === $key) {
            return $_GET;
        }

        return (Tools::getIsset($key)) ? Tools::getValue($key) : $default;
    }

    /**
     * Retrieve a member of the $_POST superglobal
     *
     * If no $key is passed, returns the entire $_POST array.
     *
     * @param string $key
     * @param mixed $default Default value to use if key not found
     * @return mixed Returns null if key does not exist
     * @todo How to retrieve from nested arrays
     */
    public function getPost($key = null, $default = null)
    {
        if (null === $key) {
            return $_POST;
        }

        return (Tools::getIsset($key)) ? Tools::getValue($key) : $default;
    }

    /**
     * Retrieve a member of the Context::getContext()->cookie superglobal
     *
     * If no $key is passed, returns the entire Context::getContext()->cookie array.
     *
     * @param string $key
     * @param mixed $default Default value to use if key not found
     * @return mixed Returns null if key does not exist
     * @todo How to retrieve from nested arrays
     */
    public function getCookie($key = null, $default = null)
    {
        if (null === $key) {
            return Context::getContext()->cookie;
        }

        return (!is_null(Context::getContext()->cookie[$key]) &&
            array_key_exists($key, Context::getContext()->cookie)) ? Context::getContext()->cookie[$key] : $default;
    }

    /**
     * Retrieve a member of the $_ENV superglobal
     *
     * If no $key is passed, returns the entire $_ENV array.
     *
     * @param string $key
     * @param mixed $default Default value to use if key not found
     * @return mixed Returns null if key does not exist
     */
    public function getEnv($key = null, $default = null)
    {
        if (null === $key) {
            return $_ENV;
        }

        return ((array_key_exists($key, $_ENV) && $_ENV[$key]) ? $_ENV[$key] : $default);
    }

    /**
     * Everything in REQUEST_URI before PATH_INFO not including the filename
     * <img src="<?=$basePath?>/images/zend.png"/>
     *
     * @return string
     */
    public function getBasePath()
    {
        if (null === $this->basePath) {
            $this->setBasePath();
        }

        return $this->basePath;
    }

    /**
     * Set the base path for the URL
     *
     * @param string|null $basePath
     * @return SZendControllerRequestHttp
     */
    public function setBasePath($basePath = null)
    {
        if ($basePath === null) {
            $filename = basename($_SERVER['SCRIPT_FILENAME']);

            $baseUrl = $this->getBaseUrl();
            if (empty($baseUrl)) {
                $this->basePath = '';

                return $this;
            }

            if (basename($baseUrl) === $filename) {
                $basePath = dirname($baseUrl);
            } else {
                $basePath = $baseUrl;
            }
        }

        $this->basePath = rtrim($basePath, '/');

        return $this;
    }

    /**
     * Retrieve a parameter
     *
     * Retrieves a parameter from the instance. Priority is in the order of
     * userland parameters (see {@link setParam()}), $_GET, $_POST. If a
     * parameter matching the $key is not found, null is returned.
     *
     * If the $key is an alias, the actual key aliased will be used.
     *
     * @param mixed $key
     * @param mixed $default Default value to use if key not found
     * @return mixed
     */
    public function getParam($key, $default = null)
    {
        $keyName = (null !== ($alias = $this->getAlias($key))) ? $alias : $key;

        $paramSources = $this->getParamSources();
        if (array_key_exists($keyName, $this->params) &&
            $this->params[$keyName]) {
            return $this->params[$keyName];
        } elseif ((in_array('_GET', $paramSources) ||
                in_array('_POST', $paramSources)) &&
            (Tools::getIsset($keyName))) {
            return Tools::getValue($keyName);
        }

        return $default;
    }

    /**
     * Retrieve an alias
     *
     * Retrieve the actual key represented by the alias $name.
     *
     * @param string $name
     * @return string|null Returns null when no alias exists
     */
    public function getAlias($name)
    {
        if (array_key_exists($name, $this->aliases) && $this->aliases[$name]) {
            return $this->aliases[$name];
        }

        return null;
    }

    /**
     * Get list of allowed parameter sources
     *
     * @return array
     */
    public function getParamSources()
    {
        return $this->paramSources;
    }

    /**
     * Set allowed parameter sources
     *
     * Can be empty array, or contain one or more of '_GET' or '_POST'.
     *
     * @param array $paramSoures
     * @return SZendControllerRequestHttp
     */
    public function setParamSources(array $paramSources = array())
    {
        $this->paramSources = $paramSources;

        return $this;
    }

    /**
     * Retrieve an array of parameters
     *
     * Retrieves a merged array of parameters, with precedence of userland
     * params (see {@link setParam()}), $_GET, $POST (i.e., values in the
     * userland params will take precedence over all others).
     *
     * @return array
     */
    public function getParams()
    {
        $return = $this->params;

        if (($_GET) && is_array($_GET)) {
            $return += $_GET;
        }
        if (($_POST) && is_array($_POST)) {
            $return += $_POST;
        }


        return $return;
    }

    /**
     * Set parameters
     *
     * Set one or more parameters. Parameters are set as userland parameters,
     * using the keys specified in the array.
     *
     * @param array $params
     * @return SZendControllerRequestHttp
     */
    public function setParams(array $params)
    {
        foreach ($params as $key => $value) {
            $this->setParam($key, $value);
        }

        return $this;
    }

    /**
     * Set a userland parameter
     *
     * Uses $key to set a userland parameter. If $key is an alias, the actual
     * key will be retrieved and used to set the parameter.
     *
     * @param mixed $key
     * @param mixed $value
     * @return SZendControllerRequestHttp
     */
    public function setParam($key, $value)
    {
        $key = (null !== ($alias = $this->getAlias($key))) ? $alias : $key;
        parent::setParam($key, $value);

        return $this;
    }

    /**
     * Set a key alias
     *
     * Set an alias used for key lookups. $name specifies the alias, $target
     * specifies the actual key to use.
     *
     * @param string $name
     * @param string $target
     * @return SZendControllerRequestHttp
     */
    public function setAlias($name, $target)
    {
        $this->aliases[$name] = $target;

        return $this;
    }

    /**
     * Retrieve the list of all aliases
     *
     * @return array
     */
    public function getAliases()
    {
        return $this->aliases;
    }

    /**
     * Was the request made by POST?
     *
     * @return boolean
     */
    public function isPost()
    {
        if ('POST' == $this->getMethod()) {
            return true;
        }

        return false;
    }

    /**
     * Return the method by which the request was made
     *
     * @return string
     */
    public function getMethod()
    {
        return $this->getServer('REQUEST_METHOD');
    }

    /**
     * Retrieve a member of the $_SERVER superglobal
     *
     * If no $key is passed, returns the entire $_SERVER array.
     *
     * @param string $key
     * @param mixed $default Default value to use if key not found
     * @return mixed Returns null if key does not exist
     */
    public function getServer($key = null, $default = null)
    {
        if (null === $key) {
            return $_SERVER;
        }
        return array_key_exists($key, $_SERVER) ? $_SERVER[$key] : $default;
    }

    /**
     * Was the request made by GET?
     *
     * @return boolean
     */
    public function isGet()
    {
        if ('GET' == $this->getMethod()) {
            return true;
        }

        return false;
    }

    /**
     * Was the request made by PUT?
     *
     * @return boolean
     */
    public function isPut()
    {
        if ('PUT' == $this->getMethod()) {
            return true;
        }

        return false;
    }

    /**
     * Was the request made by DELETE?
     *
     * @return boolean
     */
    public function isDelete()
    {
        if ('DELETE' == $this->getMethod()) {
            return true;
        }

        return false;
    }

    /**
     * Was the request made by HEAD?
     *
     * @return boolean
     */
    public function isHead()
    {
        if ('HEAD' == $this->getMethod()) {
            return true;
        }

        return false;
    }

    /**
     * Was the request made by OPTIONS?
     *
     * @return boolean
     */
    public function isOptions()
    {
        if ('OPTIONS' == $this->getMethod()) {
            return true;
        }

        return false;
    }

    /**
     * Return the raw body of the request, if present
     *
     * @return string|false Raw body, or false if not present
     */
    public function getRawBody()
    {
        $body = Tools::file_get_contents('php://input');

        if (Tools::strlen(trim($body)) > 0) {
            return $body;
        }

        return false;
    }

    /**
     * Is the request a Javascript XMLHttpRequest?
     *
     * Should work with Prototype/Script.aculo.us, possibly others.
     *
     * @return boolean
     */
    public function isXmlHttpRequest()
    {
        return ($this->getHeader('X_REQUESTED_WITH') == 'XMLHttpRequest');
    }

    /**
     * Return the value of the given HTTP header. Pass the header name as the
     * plain, HTTP-specified header name. Ex.: Ask for 'Accept' to get the
     * Accept header, 'Accept-Encoding' to get the Accept-Encoding header.
     *
     * @param string $header HTTP header name
     * @return string|false HTTP header value, or false if not found
     * @throws SZendControllerRequestException
     */
    public function getHeader($header)
    {
        if (empty($header)) {
            require_once 'SZend/Controller/Request/Exception.php';
            throw new SZendControllerRequestException('An HTTP header name is required');
        }

        // Try to get it from the $_SERVER array first
        $temp = 'HTTP_'.Tools::strtoupper(str_replace('-', '_', $header));
        if (!empty($_SERVER[$temp])) {
            return $_SERVER[$temp];
        }

        // This seems to be the only way to get the Authorization header on
        // Apache
        if (function_exists('apache_request_headers')) {
            $headers = apache_request_headers();
            if (!empty($headers[$header])) {
                return $headers[$header];
            }
        }

        return false;
    }

    /**
     * Is this a Flash request?
     *
     * @return bool
     */
    public function isFlashRequest()
    {
        return ($this->getHeader('USER_AGENT') == 'Shockwave Flash');
    }
}
