<?php
/**
 * Server
 *
 * @author      Shopimind <contact@shopimind.com>
 * @copyright   Copyright (c) 2013 - IDVIVE SARL (http://www.idvive.com)
 * @license     New BSD license (http://license.idvive.com)
 * @package     ShopimindClient
 * @version     $Id Server.php 2013-04-23$
 */

require_once dirname(__FILE__).'/library/SZend/Controller/Request/Http.php';
require_once dirname(__FILE__).'/library/SZend/Json.php';
require_once dirname(__FILE__).'/../bin/Configuration.php';

final class ShopimindClientServer
{

    /**
     * Requête http
     *
     * @var SZendControllerRequestHttp
     */
    private $request;

    public function __construct()
    {
        $this->initRequest();
    }

    private function initRequest()
    {
        $this->setRequest(new SZendControllerRequestHttp);
    }


    private function customHmac($algo, $data, $key, $raw_output = false)
    {
        $algo = Tools::strtolower($algo);
        $pack = 'H'.Tools::strlen(hash($algo, 'test'));
        $size = 64;
        $opad = str_repeat(chr(0x5C), $size);
        $ipad = str_repeat(chr(0x36), $size);

        if (Tools::strlen($key) > $size) {
            $key = str_pad(pack($pack, hash($algo, $key)), $size, chr(0x00));
        } else {
            $key = str_pad($key, $size, chr(0x00));
        }

        for ($i = 0; $i < Tools::strlen($key) - 1; $i++) {
            $opad[$i] = $opad[$i] ^ $key[$i];
            $ipad[$i] = $ipad[$i] ^ $key[$i];
        }

        $output = hash($algo, $opad.pack($pack, hash($algo, $ipad.$data)));

        return ($raw_output) ? pack($pack, $output) : $output;
    }

    public function isValid($data = null)
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        $client_id = $this->getRequest()->getHeader($config['header']['client_id']);
        $request_hmac = $this->getRequest()->getHeader($config['header']['hmac']);

        if ($data === null) {
            if ($this->getRequest()->isPost()) {
                $data = $this->getRequest()->getPost();
            } elseif ($this->getRequest()->isGet()) {
                $data = $this->getRequest()->getQuery();
            }
        }

        if (is_array($data)) {
            uksort($data, 'strnatcasecmp');
            $data = $this->implodeRecursive($data);
        }

        $hmac = $this->customHmac('sha256', $data, sha1($config['api']['password']));

        return (md5($hmac) === md5($request_hmac) && md5($config['api']['identifiant']) === md5($client_id));
    }

    /**
     * Permet de récupérer la requête
     *
     * @return SZendControllerRequestHttp
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * Permet de renseigner une request
     *
     * @param SZendControllerRequestHttp $request
     * @return ShopimindClientServer
     */
    public function setRequest($request)
    {
        $this->request = $request;

        return $this;
    }

    private function implodeRecursive(array $array, $glue = ';')
    {
        $glued_string = '';
        // Recursively iterates array and adds key/value to glued string
        array_walk_recursive(
            $array,
            array($this, 'formatString'),
            array('glue' => $glue, 'glued_string' => &$glued_string)
        );
        // Removes last $glue from string
        Tools::strlen($glue) > 0 and $glued_string = Tools::substr($glued_string, 0, -Tools::strlen($glue));

        // Trim ALL whitespace
        $glued_string = preg_replace("/(\s)/ixsm", '', $glued_string);

        return (string)$glued_string;
    }

    /**
     * Permet de retrouver la relance concernée par la requête
     *
     * @return string
     */
    public function retrieveRelaunch()
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        if ($this->getRequest()->isPost()) {
            $relaunch = $this->getRequest()->getPost($config['get']['relaunch']);
        } elseif ($this->getRequest()->isGet()) {
            $relaunch = $this->getRequest()->getQuery($config['get']['relaunch']);
        }
        if ($relaunch !== null) {
            return str_replace(' ', '', ucwords(str_replace('-', ' ', $relaunch)));
        }

        return null;
    }

    /**
     * Permet de retrouver les paramètres de relance
     *
     * @return array
     */
    public function retrieveParams()
    {
        $params = array();
        if ($this->getRequest()->isPost()) {
            $paramsReceive = $this->getRequest()->getPost();
        } elseif ($this->getRequest()->isGet()) {
            $paramsReceive = $this->getRequest()->getQuery();
        }
        foreach ($paramsReceive as $name => $val) {
            if (!in_array($name, $params)) {
                $params[$name] = $val;
            }
        }

        return $params;
    }

    /**
     * Permet de récupérer le type de requête que l'on souhaite executer
     *
     * @return string
     */
    public function getTypeRequest()
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        return $this->getRequest()->getHeader($config['get']['type-request']);
    }

    /**
     * Permet d'envoyer une réponse
     *
     * @param array|string $data
     * @return string
     */
    public function sendResponse(array $data, $success = true)
    {
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
        header('Content-type: application/json');


        if (!array_key_exists('success', $data)) {
            $data['success'] = $success;
        }

        echo SZendJson::encode($data);
        exit;
    }

    protected function formatString($value, $key, $options)
    {
        if (!empty($value)) {
            $options['glued_string'] .= $key.$options['glue'];
            $options['glued_string'] .= $value.$options['glue'];
        }
    }
}
