<?php
/**
 * Store Commander
 *
 * @author Store Commander - support@storecommander.com
 * @copyright Copyright &copy; 2009-2022, Store Commander
 * @license commercial
 * @category administration
 * @version 1.2.0
 * @uses Prestashop modules
 * @since 2009
 * All rights reserved! Copying, duplication strictly prohibited
 *
 * *****************************************
 * *           STORE COMMANDER             *
 * *   https://www.storecommander.com      *
 * *            V 1.2.0       *
 * *****************************************
 *
 * Compatibility: PS version: 1.6 to 1.7
 *
 **/
if (!defined('_PS_VERSION_')) {
    exit;
}
require __DIR__.'/vendor/autoload.php';

/**
 *
 */
class StoreCommanderPs extends Module
{

    const STEP_READY_TO_INSTALL = 1;
    const STEP_START_INSTALL = 2;
    const STEP_INSTALLED = 3;
    const STEP_WAIT_FOR_LICENSE = 4;

    /**
     * @var array
     */
    public $_err = [];
    /**
     * @var string
     */
    private $url_zip_SC = "https://www.storecommander.com/files/StoreCommander.zip";
    /**
     * @var string
     */
    private $url_api = "https://api.storecommander.com/rbmSubscription/check/";
    /**
     * @var ServiceContainer
     */
    private $container;
    /**
     * @var bool
     */
    private $psVersionIs17;
    /**
     * @var string
     */
    private $emailSupport;
    /**
     * @var
     */
    private $isPsAccountLinked;
    /**
     * @var
     */
    private $userEmail;
    /**
     * @var
     */
    private $shopUrl;
    /**
     * @var
     */
    private $userId;

    private $sc_allowed_iso = ['en','es','fr'];

    private $reset_keep_folder = null;

    public function __construct()
    {
        $this->name = 'storecommanderps';
        $this->tab = 'administration';
        $this->version = '1.2.0';
        $this->author = 'Store Commander';
        $this->emailSupport = 'support@storecommander.com';
        $this->module_key = 'fe3810472eabee62dccf29a9f10fc703';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.6.1.0',
            'max' => _PS_VERSION_,
        ];
        $this->psVersionIs17 = (bool)version_compare(_PS_VERSION_, '1.7', '>=');
        $this->bootstrap = true;

        parent::__construct();

        $this->page = basename(__FILE__, '.php');
        $this->displayName = $this->author;
        $this->description = $this->l('Boost your backoffice management! Activate and manage your Store Commander subscription directly from your PrestaShop admin.');
        $this->confirmUninstall = $this->l('Warning! This action definitely uninstall Store Commander!');

        $warning = [];
        if (!is_writable(_PS_ROOT_DIR_.'/modules/'.$this->name)) {
            $warning[] = $this->l('The /modules/'.$this->name.' folder must be writable.');
        }
        if (!Configuration::getGlobalValue('SC_INSTALLED')) {
            $warning[] = $this->l('Store Commander is not installed!');
        }
        if (!empty($warning)) {
            $this->warning = implode(' ', $warning);
        }

        $this->template_dir = _PS_MODULE_DIR_.$this->name.'/views/templates/admin/';

        if ($this->container === null) {
            $this->container = new \PrestaShop\ModuleLibServiceContainer\DependencyInjection\ServiceContainer($this->name, $this->getLocalPath());
        }

    }

    /**
     * @return bool
     */
    public function install()
    {
        // cohabitation storecommander et storecommanderps impossible, verification avant installation
        if($this->psVersionIs17){
            $legacyLogger = new PrestaShop\PrestaShop\Adapter\LegacyLogger();
            $moduleDataProvider = new PrestaShop\PrestaShop\Adapter\Module\ModuleDataProvider($legacyLogger, $this->getTranslator());
            $module_installed =$moduleDataProvider->isInstalled('storecommander');
        } else {
            $module_installed = Module::isInstalled('storecommander');
        }
        if(is_dir(_PS_ROOT_DIR_.'/modules/storecommander') && $module_installed){
            $this->_errors[] = $this->l('Store Commander is already installed on your shop. You can launch the application from the "Modules > Store Commander" menu. Please contact us at support@storecommander.com for further information.');
            return false;
        }

        $sc_folder_hash = (!empty($this->reset_keep_folder) ? $this->reset_keep_folder : Tools::substr(md5(date("YmdHis")._COOKIE_KEY_), 0, 11));

        if (!parent::install()
            || !Configuration::updateGlobalValue('SC_FOLDER_HASH', $sc_folder_hash)
            || !$this->createSCFolder($sc_folder_hash)
            || !Configuration::updateGlobalValue('SC_INSTALLED', false)
            || !$this->registerHook('displayBackOfficeFooter')
            || !$this->getService('ps_accounts.installer')->install()
        ) {
            return false;
        }

        return true;
    }

    /**
     * @param $folder
     * @return bool|void
     */
    private function createSCFolder($folder)
    {
        if (!is_dir(dirname(__FILE__).'/'.$folder)) {
            return mkdir(dirname(__FILE__).'/'.$folder);
        }
    }

    /**
     * Retrieve service
     *
     * @param string $serviceName
     *
     * @return mixed
     */
    public function getService($serviceName)
    {
        if ($this->container === null) {
            $this->container = new \PrestaShop\ModuleLibServiceContainer\DependencyInjection\ServiceContainer(
                $this->name,
                $this->getLocalPath()
            );
        }

        return $this->container->getService($serviceName);
    }

    // SC specific functions

    /**
     * @return bool
     */
    public function uninstall()
    {
        $this->reset_keep_folder = Configuration::getGlobalValue('SC_FOLDER_HASH');

        try {
            $qaccess = Db::getInstance()->getValue(
                "SELECT GROUP_CONCAT(`id_quick_access`) AS qaccess FROM `"._DB_PREFIX_.
                "quick_access` WHERE `link` LIKE '%".$this->name."%'"
            );
            if ($qaccess != '') {
                Db::getInstance()->Execute(
                    "DELETE FROM `"._DB_PREFIX_."quick_access` WHERE id_quick_access IN (".psql($qaccess).")"
                );
                Db::getInstance()->Execute(
                    "DELETE FROM `"._DB_PREFIX_."quick_access_lang` WHERE id_quick_access IN (".psql($qaccess).")"
                );
            }
            $tab = new Tab(Tab::getIdFromClassName('AdminStoreCommanderPs'));
            $tab->delete();
            $this->removeSCFolder($this->reset_keep_folder);
            Configuration::deleteByName('SC_FOLDER_HASH');
            Configuration::deleteByName('SC_INSTALLED');
            Configuration::deleteByName('SC_SETTINGS');
            Configuration::deleteByName('SC_LICENSE_DATA');
            Configuration::deleteByName('SC_LICENSE_KEY');
            Configuration::deleteByName('SC_VERSIONS');
            Configuration::deleteByName('SC_VERSIONS_LAST');
            Configuration::deleteByName('SC_VERSIONS_LAST_CHECK');

            parent::uninstall();

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * @param $folder
     * @return bool
     */
    private function removeSCFolder($folder)
    {
        if (is_dir(dirname(__FILE__).'/'.$folder)) {
            $this->rrmdir(dirname(__FILE__).'/'.$folder);
        }

        return true;
    }

    /**
     * @param $dir
     * @return bool
     */
    private function rrmdir($dir)
    {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (filetype($dir."/".$object) == "dir") {
                        $this->rrmdir($dir."/".$object);
                    } else {
                        unlink($dir."/".$object);
                    }
                }
            }
            reset($objects);
            @rmdir($dir);
        }

        return true;
    }

    public function getContent()
    {
        $scStepParam = (int)Tools::getValue("sc_step");
        $guessedStep = $this->guessStep($scStepParam);
        $urlArray = parse_url(_PS_BASE_URL_);
        $this->shopUrl = $urlArray['host'];

        if ($guessedStep != $scStepParam && !Tools::getValue('ajax', false)) {
            Tools::redirectAdmin(
                $this->getScAdminLink([
                    'sc_step' => $guessedStep,
                ])
            );
        }

        $facade = $this->getService('ps_accounts.facade');
        Media::addJsDef([
            'contextPsAccounts' => $facade->getPsAccountsPresenter()->present($this->name),
        ]);

        $this->context->smarty->assign([
            'pathSettingsVendor' => $this->getPathUri().'views/js/'.$this->getLastJsFileName('chunk-vendors-*.js'),
            'pathSettingsApp' => $this->getPathUri().'views/js/'.$this->getLastJsFileName('app-*.js')
        ]);

        try {
            $psAccountsService = $facade->getPsAccountsService();
            $this->userEmail = $psAccountsService->getEmail();
            $this->userId = (method_exists($psAccountsService, 'getUserUuid') ? $psAccountsService->getUserUuid() : $psAccountsService->getUserUuidV4());
            $shopUuid = (method_exists($psAccountsService, 'getShopUuid') ? $psAccountsService->getShopUuid() : $psAccountsService->getShopUuidV4());
            $refreshToken = $psAccountsService->getRefreshToken();

            $this->isPsAccountLinked = ($psAccountsService->isAccountLinked() ? $psAccountsService->isAccountLinked() : $psAccountsService->isAccountLinkedV4());

            if ($guessedStep === self::STEP_WAIT_FOR_LICENSE || Tools::getValue('checkLicense', false)) {
                if (Tools::getValue('checkLicense', false)) {
                    exit($this->checkLicense());
                }
            }
            
            Media::addJsDef([
                'psBillingContext' => [
                    'context' => [
                        'isMS' => Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE'),
                        'versionPs' => _PS_VERSION_,
                        'versionModule' => $this->version,
                        'moduleName' => $this->name,
                        'accountApi' => $psAccountsService->getAdminAjaxUrl(),
                        'emailSupport' => $this->emailSupport,
                        'refreshToken' => $refreshToken,
                        'moduleTosUrl' => $this->getTosLink(),
                        'i18n' => [
                            'isoCode' => $this->getLanguageIsoCode(),
                        ],
                        'shop' => [
                            'uuid' => $shopUuid,
                        ],
                        'user' => [
                            'createdFromIp' => (isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : ''),
                            'isPsAccountLinked' => (bool)$this->isPsAccountLinked,
                            'email' => $this->userEmail,
                        ],
                        'moduleMessages' => $this->getModuleMessages()
                    ],
                ],
            ]);

        } catch (Exception $e) {
            $this->context->controller->errors[] = $e->getMessage();
            return '';
        }


        $this->context->controller->addJS(__PS_BASE_URI__.'modules/'.$this->name.'/views/js/loader/jquery.loader-min.js');
        $this->context->controller->addJS("https://js.chargebee.com/v2/chargebee.js");
        $this->context->controller->addCSS(__PS_BASE_URI__.'modules/'.$this->name.'/views/css/admin.css?v='.$this->version, 'all', null,false);

        $this->context->smarty->assign([
            'scBanner' => __PS_BASE_URI__.'modules/'.$this->name.'/views/img/marketing/banner_sc_'.$this->getAllowedIsoCodeForSc().'.png',
            'scPanelContent' => $this->displayStep($guessedStep),
            'urlAccountsVueCdn' => $psAccountsService->getAccountsVueCdn(),
            'scAllowedIsoCode' => $this->getAllowedIsoCodeForSc()
        ]);

        return $this->context->smarty->fetch($this->template_dir.'storeCommanderPsSettings.tpl');

    }

    /**
     * @param int $step
     * @return int|mixed
     */
    private function guessStep($step = self::STEP_READY_TO_INSTALL)
    {
        if ($step === self::STEP_START_INSTALL) {
            return $step;
        }

        $licenseKey = Configuration::getGlobalValue('SC_LICENSE_KEY', false);
        $scInstalled = Configuration::getGlobalValue('SC_INSTALLED') && $this->isSCFolderReady();
        if ($scInstalled) {
            $step = self::STEP_INSTALLED;
            if (!$licenseKey) {
                $step = self::STEP_WAIT_FOR_LICENSE;
            }
        } else {
            $step = self::STEP_READY_TO_INSTALL;
        }

        return $step;
    }

    /**
     * @return bool
     */
    public function isSCFolderReady()
    {
        if (file_exists(__DIR__.'/'.Configuration::getGlobalValue('SC_FOLDER_HASH').'/SC/index.php')) {
            return true;
        }

        return false;
    }

    /**
     * @param $params
     * @return string
     */
    public function getScAdminLink($params)
    {
        $token = Tools::getValue("token", "");
        $baseParams = "?controller=AdminModules&configure=".$this->name."&token=".$token;
        $current_admin_shop_url = Tools::getShopProtocol().Tools::getShopDomainSsl().__PS_BASE_URI__.basename(_PS_ADMIN_DIR_);
        return $current_admin_shop_url.'/'.$baseParams.'&'.http_build_query($params);
    }

    /**
     * @return false
     */
    private function checkLicense()
    {
        if (!empty(Configuration::getGlobalValue('SC_LICENSE_KEY'))) {
            return Configuration::getGlobalValue('SC_LICENSE_KEY');
        }

        $hash = base64_encode(
            implode('|', [
                $this->userId,
                $this->shopUrl,
                __DIR__.'/'.Configuration::getGlobalValue('SC_FOLDER_HASH'),
                $this->getLanguageIsoCode()
            ])
        );

        $response = Tools::file_get_contents($this->url_api.$hash,false,null,30);
        if(!$response) {
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl, CURLOPT_URL, $this->url_api.$hash);
            curl_setopt($curl, CURLOPT_HEADER, 0);
            curl_setopt($curl, CURLOPT_TIMEOUT, 30);
            $response = curl_exec($curl);
            curl_close($curl);
        }
        if(!empty($response)) {
            $result = json_decode($response, true);
            if (is_array($result)
                && !empty($result)
                && array_key_exists('license', $result)
                && $result['license'] !== '') {
                $license = (string)$result['license'];
                Configuration::updateGlobalValue('SC_LICENSE_KEY', $license);
                return $license;
            }
        }
        return false;
    }

    /**
     * @return string
     */
    public function getLanguageIsoCode()
    {
        return $this->context->language !== null ? $this->context->language->iso_code : 'en';
    }
    /**
     * @return string
     */
    public function getAllowedIsoCodeForSc()
    {
        $iso = strtolower($this->getLanguageIsoCode());
        return in_array($iso,$this->sc_allowed_iso) ? $iso : 'en';
    }

    /**
     * @return string
     */
    public function getTosLink()
    {
        $iso_lang = $this->getLanguageIsoCode();
        switch ($iso_lang) {
            case 'fr':
                $url = 'https://www.storecommander.com/fr/content/84-conditions-generales-de-vente-et-d-utilisation-de-store-commander-rbm';
                break;
            case 'es':
                $url = 'https://www.storecommander.com/es/content/84-condiciones-generales-de-venta-y-uso-rbm';
                break;
            default:
                $url = 'https://www.storecommander.com/en/content/84-terms-and-conditions-of-use-rbm';
                break;
        }

        return $url;
    }

    /**
     * @return array
     */
    public function getModuleMessages()
    {
        $messages = [];
        $messages['error_plan'] = $this->l('You cannot use a solo or solo + plan on a multistore website. Please choose at least multi plan');
        $messages['warning_plan'] = $this->l('Warning ! You have selected a Solo plan which will not allow you to continue to use Store Commander in MultiStore');

        return $messages;
    }

    /**
     * @param $step
     * @return string|void
     */
    private function displayStep($step)
    {
        // pas de panel SC si pas de license et compte non lié
        if (!$this->isPsAccountLinked) {
            return '';
        }

        switch ((int)$step) {
            case self::STEP_READY_TO_INSTALL : // affichage panel installation ?
                $this->context->smarty->assign([
                    'linkToConfigureModule' => $this->getScAdminLink(["sc_step" => self::STEP_START_INSTALL]),
                ]);

                return $this->display(__FILE__, 'etape_preinstall.tpl');
            case self::STEP_START_INSTALL : // téléchargement sc
                if (!$this->downloadExtractSC()) {
                    $this->context->controller->errors[] = $this->l('Error while downloading Store Commander');
                } else {
                    Configuration::updateGlobalValue('SC_INSTALLED', true);
                    if (file_exists(__DIR__.'/license.php')) {
                        @copy(
                            __DIR__.'/license.php',
                            _PS_MODULE_DIR_.$this->name.'/'.Configuration::getGlobalValue('SC_FOLDER_HASH').
                            '/SC/license.php'
                        );
                    }
                    Tools::redirectAdmin($this->getScAdminLink(['sc_step' => self::STEP_WAIT_FOR_LICENSE]));
                }
                break;
            case self::STEP_INSTALLED : // panel module installé + Update
                $this->createTab();
                $update_validation = (int)Tools::getValue("SCupdateValidation");
                $SCupdate = Tools::getValue("SCupdate");
                if (!empty($SCupdate)) {
                    $this->makeSCupdate();
                }

                $current_iso = $this->getLanguageIsoCode();
                $cgu_iso = (in_array($current_iso,$this->sc_allowed_iso) ? $current_iso : 'en');

                $token = Tools::getAdminToken(
                    'AdminStoreCommanderPs'.(int)Tab::getIdFromClassName('AdminStoreCommanderPs').
                    (int)($this->context->employee->id)
                );

                $linkToStoreCommander = "index.php?controller=AdminStoreCommanderPs&token=".$token;
                $updateLink = $this->getScAdminLink([
                    "sc_step" => self::STEP_INSTALLED,
                    'SCupdate' => 1,
                ]);

                $this->context->smarty->assign([
                    'cgu_iso' => $cgu_iso,
                    'token' => $token,
                    'linkToStoreCommander' => $linkToStoreCommander,
                    'updateLink' => $updateLink,
                    'update_validation' => $update_validation,
                ]);

                return $this->display(__FILE__, 'etape_postinstall.tpl');
            case self::STEP_WAIT_FOR_LICENSE : //attente de recuperation de la clé de licence
                $this->context->smarty->assign([
                    'scCheckLicenseLink' => $this->getScAdminLink(['checkLicense' => true, 'ajax' => 1]),
                ]);
                if (Configuration::getGlobalValue('SC_LICENSE_KEY')) {
                    Tools::redirectAdmin($this->getScAdminLink(['sc_step' => self::STEP_INSTALLED]));
                } else {
                    return $this->display(__FILE__, 'etape_waitlicense.tpl');
                }
                break;
            default :
                return '';
        }
    }

    //step

    /**
     * @return bool
     */
    private function downloadExtractSC()
    {
        $data = $this->scFileGetContents($this->url_zip_SC);
        file_put_contents(_PS_MODULE_DIR_.$this->name.'/'.basename($this->url_zip_SC), $data);

        return $this->extractArchive(_PS_MODULE_DIR_.$this->name.'/'.basename($this->url_zip_SC));
    }

    /**
     * @param $param
     * @return bool|string
     */
    public function scFileGetContents($param)
    {
        @$result = Tools::file_get_contents($param, false, null, 30);

        if ($result == '' && function_exists('curl_init')) {
            $curl = curl_init();
            $header = [];
            $header[0] = "Accept: text/xml,application/xml,application/xhtml+xml,";
            $header[0] = $header[0]."text/html;q=0.9,text/plain;q=0.8,image/png,*/*;q=0.5";
            $header[] = "Cache-Control: max-age=0";
            $header[] = "Connection: keep-alive";
            $header[] = "Keep-Alive: 300";
            $header[] = "Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7";
            $header[] = "Accept-Language: en-us,en;q=0.5";
            $header[] = "Pragma: ";
            curl_setopt($curl, CURLOPT_URL, $param);
            curl_setopt($curl, CURLOPT_USERAGENT, 'Store Commander (https://www.storecommander.com)');
            curl_setopt($curl, CURLOPT_HTTPHEADER, $header);
            curl_setopt($curl, CURLOPT_ENCODING, 'gzip,deflate');
            curl_setopt($curl, CURLOPT_AUTOREFERER, true);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($curl, CURLOPT_TIMEOUT, 20);
            $result = curl_exec($curl);
            $info = curl_getinfo($curl);
            curl_close($curl);
            if ((int)$info['http_code'] != 200) {
                return '';
            }
        }

        return $result;
    }

    /**
     * @param $file
     * @param false $noPCLZIP
     * @return bool
     */
    private function extractArchive($file, $noPCLZIP = false)
    {
        $success = true;
        if (file_exists(_PS_TOOL_DIR_.'pclzip/pclzip.lib.php') && empty($noPCLZIP)) {
            require_once(_PS_TOOL_DIR_.'pclzip/pclzip.lib.php');
            $zip = new PclZip($file);
            $list = $zip->extract(
                PCLZIP_OPT_PATH,
                _PS_MODULE_DIR_.$this->name.'/'.Configuration::getGlobalValue('SC_FOLDER_HASH')
            );
            foreach ($list as $extractedFile) {
                if ($extractedFile['status'] != 'ok') {
                    $success = false;
                }
            }
        } else {
            if (class_exists('ZipArchive', false)) {
                $zip = new ZipArchive();
                if ($zip->open($file) === true and
                    $zip->extractTo(_PS_MODULE_DIR_.$this->name.'/'.Configuration::getGlobalValue('SC_FOLDER_HASH')) and
                    $zip->close()) {
                    $success = true;
                } else {
                    $success = false;
                }
            } else {
                $success = false;
            }
        }
        @unlink($file);

        return $success;
    }


    // RBM functions

    // common useful functions


    private function createTab()
    {
        if($this->psVersionIs17){
            $parentTabName = 'AdminParentModulesSf';
        } else {
            $parentTabName = 'AdminParentModules';
        }
        $tab = new Tab(Tab::getIdFromClassName('AdminStoreCommanderPs'));
        $tab->class_name = 'AdminStoreCommanderPs';
        $tab->id_parent = (int)Tab::getIdFromClassName($parentTabName);
        $tab->active = 1;
        $tab->module = $this->name;
        foreach (Language::getLanguages(false) as $language) {
            $tab->name[$language["id_lang"]] = 'Store Commander';
        }
        $tabSaved = $tab->save();
        if(!$tabSaved){
            $sql = 'SELECT id_quick_access AS id FROM `' . _DB_PREFIX_ . 'quick_access` q WHERE q.`link` LIKE \'%AdminStoreCommanderPs%\'';
            $result = Db::getInstance()->getRow($sql);
            if (!$result) {
                $quickAccess = new QuickAccess();
                $tmp = array();
                $languages = Language::getLanguages();
                foreach ($languages AS $lang) {
                    $tmp[$lang['id_lang']] = "Store Commander";
                }
                $quickAccess->name = $tmp;
                $quickAccess->link = "index.php?controller=AdminStoreCommanderPs";
                $quickAccess->new_window = true;
                $quickAccess->add();
            }
        }
    }

    /**
     * @return string
     */
    private function makeSCupdate()
    {
        $data = $this->scFileGetContents($this->url_zip_SC);
        file_put_contents(_PS_MODULE_DIR_.$this->name.'/'.basename($this->url_zip_SC), $data);
        $var = $this->extractArchive(_PS_MODULE_DIR_.$this->name.'/'.basename($this->url_zip_SC), true);
        if (!$var) {
            $this->context->controller->errors[] =  $this->l('Error while downloading Store Commander');
        } else {
            Tools::redirectAdmin(
                $this->getScAdminLink([
                    'sc_step' => self::STEP_INSTALLED,
                    'SCupdate' => 0,
                    'SCupdateValidation' => 1,
                ])
            );

        }
    }

    public function hookBackOfficeFooter()
    {
        $is_completly_installed = (Configuration::getGlobalValue('SC_LICENSE_KEY', false) && Configuration::getGlobalValue('SC_INSTALLED') && $this->isSCFolderReady());
        if($is_completly_installed) {
            $this->displayBackOfficeMessage();
        }
    }

    private function displayBackOfficeMessage(){
        $js_action = null;
        $js_id = null;
        if (version_compare(_PS_VERSION_, '1.6.1.0', '>=')) {
            $cookie = $this->context->cookie;
        } else {
            $cookie = new Cookie();
        }
        $show_msg = 1;
        if (version_compare(_PS_VERSION_, '1.7.7.0', '>=')) {
            $controller = '';
            switch (get_class($this->context->controller)) {
                case 'AdminImagesController':
                    $controller = 'image';
                    $js_action = '$(".alert.alert-info").last().after('.
                        $this->getMessage($controller, 177, ['class' => 'alert alert-danger']).')';
                    break;
            }
            switch ($this->context->controller->php_self) {
                case 'AdminOrders':
                    $js_id = '#order_grid_panel';
                    $controller = 'order';
                    break;
                case 'AdminCustomers':
                    $js_id = '#customer_grid_panel';
                    $controller = 'customer';
                    break;
                case 'AdminProducts':
                    $js_id = '#main-div > .content-div > .row > div > .products-catalog';
                    $controller = 'catalog';
                    break;
                case 'AdminCategories':
                    $js_id = '#category_grid_panel';
                    $controller = 'catalog';
                    break;
            }
            if (empty($js_action) && !empty($js_id) && !empty($controller)) {
                $js_action = '$("'.$js_id.'").before('.$this->getMessage($controller, 177).');';
            }
        } else {
            if (version_compare(_PS_VERSION_, '1.7.6.5', '>=')) {
                $controller = '';
                switch (get_class($this->context->controller)) {
                    case 'AdminOrdersController':
                        $js_id = '#order-empty-filters-alert';
                        $controller = 'order';
                        break;
                    case 'AdminImagesController':
                        $controller = 'image';
                        $js_action = '$(".alert.alert-info").last().after('.
                            $this->getMessage($controller, 176, ['class' => 'alert alert-danger']).')';
                        break;
                }
                switch ($this->context->controller->php_self) {
                    case 'AdminProducts':
                        $js_id = '#main-div > .content-div > .row > div > .products-catalog';
                        $controller = 'catalog';
                        break;
                    case 'AdminCategories':
                        $js_id = '#category_grid_panel';
                        $controller = 'catalog';
                        break;
                    case 'AdminCustomers':
                        $js_id = '#customer_grid_panel';
                        $controller = 'customer';
                        break;
                }
                if (empty($js_action) && !empty($js_id) && !empty($controller)) {
                    $js_action = '$("'.$js_id.'").before('.$this->getMessage($controller, 176).');';
                }
            } else {
                if (version_compare(_PS_VERSION_, '1.7.6.0', '>=')) {
                    $controller = '';
                    switch (get_class($this->context->controller)) {
                        case 'AdminOrdersController':
                            $js_id = '#order-empty-filters-alert';
                            $controller = 'order';
                            break;
                        case 'AdminCustomersController':
                            $js_id = '#customer-empty-filters-alert';
                            $controller = 'customer';
                            break;
                        case 'AdminImagesController':
                            $controller = 'image';
                            $js_action = '$(".alert.alert-info").last().after('.
                                $this->getMessage($controller, 176, ['class' => 'alert alert-danger']).')';
                            break;
                    }
                    switch ($this->context->controller->php_self) {
                        case 'AdminProducts':
                            $js_id = '#main-div > .content-div > .row > div > .products-catalog';
                            $controller = 'catalog';
                            break;
                        case 'AdminCategories':
                            $js_id = '#category_grid_panel';
                            $controller = 'catalog';
                            break;
                    }
                    if (empty($js_action) && !empty($js_id) && !empty($controller)) {
                        $js_action = '$("'.$js_id.'").before('.$this->getMessage($controller, 176).');';
                    }
                } else {
                    if (version_compare(_PS_VERSION_, '1.7.5.0', '>=')) {
                        $controller = '';
                        switch (get_class($this->context->controller)) {
                            case 'AdminOrdersController':
                                $js_id = '#order-empty-filters-alert';
                                $controller = 'order';
                                break;
                            case 'AdminCustomersController':
                                $js_id = '#customer-empty-filters-alert';
                                $controller = 'customer';
                                break;
                            case 'AdminCategoriesController':
                                $js_id = '#content > .row > div > .leadin';
                                $controller = 'catalog';
                                break;
                            case 'AdminImagesController':
                                $controller = 'image';
                                $js_action = '$(".alert.alert-info").last().after('.
                                    $this->getMessage($controller, 175, ['class' => 'alert alert-danger']).')';
                                break;
                        }
                        switch ($this->context->controller->php_self) {
                            case 'AdminProducts':
                                $js_id = '#main-div > .content-div > .row > div > .products-catalog';
                                $controller = 'catalog';
                                break;
                        }
                        if (empty($js_action) && !empty($js_id) && !empty($controller)) {
                            $js_action = '$("'.$js_id.'").before('.$this->getMessage($controller, 175).');';
                        }
                    } else {
                        if (version_compare(_PS_VERSION_, '1.6.1.0', '>=')) {
                            $controller = '';
                            switch (get_class($this->context->controller)) {
                                case 'AdminOrdersController':
                                    $js_id = '#order-empty-filters-alert';
                                    $controller = 'order';
                                    break;
                                case 'AdminCustomersController':
                                    $js_id = '#customer-empty-filters-alert';
                                    $controller = 'customer';
                                    break;
                                case 'AdminProductsController':
                                case 'AdminCategoriesController':
                                    $js_id = '#content > .row > div > .leadin';
                                    $controller = 'catalog';
                                    break;
                                case 'AdminImagesController':
                                    $controller = 'image';
                                    $js_action = '$(".alert.alert-info").last().after('.
                                        $this->getMessage($controller, 16, ['class' => 'alert alert-danger']).')';
                                    break;
                            }
                            if (empty($js_action) && !empty($js_id) && !empty($controller)) {
                                $js_action = '$("'.$js_id.'").before('.$this->getMessage($controller, 16).');';
                            }
                        } else {
                            $controller = '';
                            switch (get_class($this->context->controller)) {
                                case 'AdminOrdersController':
                                    $js_id = '#order_toolbar';
                                    $controller = 'order';
                                    break;
                                case 'AdminCustomersController':
                                    $js_id = '#customer_toolbar';
                                    $controller = 'customer';
                                    break;
                                case 'AdminProductsController':
                                    $js_id = '#product_toolbar';
                                    $controller = 'catalog';
                                    break;
                                case 'AdminCategoriesController':
                                    $js_id = '#category_toolbar';
                                    $controller = 'catalog';
                                    break;
                                case 'AdminImagesController':
                                    $controller = 'image';
                                    $js_action = '$("input[name=submitRegenerateimage_type]").parent().before('.
                                        $this->getMessage($controller, 15, ['class' => 'warn']).')';
                                    break;
                            }
                            if (empty($js_action) && !empty($js_id) && !empty($controller)) {
                                $js_action = '$("'.$js_id.'").after('.$this->getMessage($controller, 15).');';
                            }
                        }
                    }
                }
            }
        }
        if (!empty($controller)) {
            $cookie_key = 'sc_no_msg_'.$controller;
            $cookie_no_msg = $cookie->__get($cookie_key);
            if (empty($cookie_no_msg)) {
                $no_msg = Tools::getValue($cookie_key, 0);
                if ($no_msg) {
                    $cookie->__set($cookie_key, 1);
                    $show_msg = 0;
                }
            } else {
                $show_msg = 0;
            }
        }
        if (!empty($js_action) && !empty($show_msg)) {
            $html = '<script type="text/javascript">'."\n";
            $html .= '  $(document).ready(function(){'."\n";
            $html .= $js_action."\n";
            $html .= "\n".'  });'."\n";
            $html .= '</script>'."\n";
            echo $html;
        }
    }

    /**
     * @param $controller
     * @param $ps_version
     * @param null $params
     * @return false|string
     */
    private function getMessage($controller, $ps_version, $params = null)
    {
        $class = 'ps'.$ps_version;
        $html = $sc_message_style = $bo_url = $hide_msg_url = '';
        switch ($ps_version) {
            case 15:
                $class = 'hint clear';
                $sc_message_style = 'style="display:block!important;"';
                $iso_lang = $this->getLanguageIsoCode();
                $link = new Link();
                $bo_url = $link->getAdminLink('AdminStoreCommanderPs');
                $cnt = Tools::getValue('controller');
                $hide_msg_url = $link->getAdminLink($cnt);
                break;
            case 16:
            case 175:
            case 176:
            case 177:
                $class = 'alert alert-info';
                $iso_lang = $this->getLanguageIsoCode();
                $link = new Link();
                $bo_url = $link->getAdminLink('AdminStoreCommanderPs');
                $cnt = Tools::getValue('controller');
                $hide_msg_url = $link->getAdminLink($cnt);
                break;
            default:
                global $cookie;
                $class = 'hint clear';
                $sc_message_style = 'style="display:block!important;"';
                $iso_lang = Language::getIsoById((int)$cookie->id_lang);
                $bo_url = $_SERVER['SCRIPT_NAME'].'?tab=AdminStoreCommanderPsController&token='.
                    Tools::getAdminTokenLite('AdminStoreCommanderPsController');
                $hide_msg_url = $_SERVER['REQUEST_URI'];
                break;
        }
        if (!empty($params) && is_array($params)) {
            if (array_key_exists('class', $params) && !empty($params['class'])) {
                $class = (string)$params['class'];
            }
        }
        $text = [
            'order' => [
                'fr' => '<p><b>Store Commander</b> est installé sur votre boutique. Saviez-vous que vous pouvez :</p>
                        <ul style="list-style-type: disc;padding-left:40px">
                        <li>Filtrer plus finement vos dernières commandes pour changer leurs statuts en masse</li>
                        <li>Créer des commandes pour vos clients à partir d\'un simple appel téléphonique</li>
                        <li>Rechercher, trouver et imprimer les factures/avoirs/bons de livraison plus facilement</li>
                        <li>Analyser précisément l\'origine de vos commandes pour prendre des décisions sur vos différents canaux d\'acquisition</li>
                        <li>Exporter les données pour faciliter la gestion de votre comptabilité (ventillation TVA, Pays, Mini-guichet...)</li>
                        </ul><br/>
                        <p>Démarrez Store Commander pour gérer vos commandes plus efficacement, en <b><a href="'.
                    $bo_url.'">cliquant ici</a></b> ou depuis le menu Module > Store Commander</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_order=1">Ne plus afficher ce message</a></p>',
                'en' => '<p><b>Store Commander</b> is installed on your shop. Did you know that you can:</p>
                        <ul style="list-style-type: disc;padding-left:40px">
                        <li>Filter more finely your latest orders to change their status en masse</li>
                        <li>Create orders for your customers from a simple phone call</li>
                        <li>Search, find and print invoices/credit notes/delivery notes more easily</li>
                        <li>Precisely analyze the origin of your orders to make decisions on your different acquisition channels</li>
                        <li>Export data to facilitate the management of your accounting (VAT breakdown, Country, Mini-window...)</li>
                        </ul><br/>
                        <p>Start Store Commander to manage your orders more efficiently, by <b><a href="'.$bo_url.'">clicking here</a></b> or from the Module > Store Commander menu</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_order=1">Do not display this message again</a></p>',
            ],
            'customer' => [
                'fr' => '<p><b>Store Commander</b> est installé sur votre boutique. Saviez-vous que vous pouvez :</p>
                        <ul style="list-style-type: disc;padding-left:40px">
                        <li>Trouver rapidement toutes les informations se rapportant à un client</li>
                        <li>Gérer tout votre service client depuis une seule page pour plus d\'efficacité</li>
                        <li>Exporter et importer vos clients avec plus de souplesse</li>
                        <li>Segmenter vos clients en fonction de leur comportement pour des actions marketing ciblées</li>
                        <li>Se connecter en tant qu\'un client sur son compte dans la boutique pour voir ce qu\'il voit</li>
                        </ul><br/>
                        <p>Démarrez Store Commander pour gérer vos clients plus efficacement, en <b><a href="'.$bo_url.'">cliquant ici</a></b> ou depuis le menu Module > Store Commander</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_customer=1">Ne plus afficher ce message</a></p>',
                'en' => '<p><b>Store Commander</b> is installed on your shop. Did you know that you can:</p>
                        <ul style="list-style-type: disc;padding-left:40px">
                        <li>Quickly find all information related to a customer</li>
                        <li>Manage all your customer service from a single page for greater efficiency</li>
                        <li>Export and import your customers more flexibly</li>
                        <li>Segment your customers according to their behaviour for targeted marketing actions</li>
                        <li>Log in as a customer on his account in the shop to see what he sees</li>
                        </ul><br/>
                        <p>Start Store Commander to manage your customers more efficiently, by <b><a href="'.$bo_url.'">clicking here</a></b> or from the Module > Store Commander menu</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_customer=1">Do not display this message again</a></p>',
            ],
            'catalog' => [
                'fr' => '<p><b>Store Commander</b> est installé sur votre boutique. Saviez-vous que vous pouvez :</p>
                        <ul style="list-style-type: disc;padding-left:40px">
                        <li>Trouver et modifier en masse toutes les informations de vos produits</li>
                        <li>Exporter et ré-importer tout votre catalogue ou une sélection de produits</li>
                        <li>Préparer les périodes de soldes et promotions vitesse turbo, en contrôlant vos marges</li>
                        <li>Améliorer tous les critères SEO de chaque fiche produit : meta tags, taille des meta tags...</li>
                        <li>Identifier et corriger des centaines de problèmes avec le menu Outils > FixMyPrestaShop</li>
                        </ul><br/>
                        <p>Démarrez Store Commander pour gérer votre catalogue plus efficacement, en <b><a href="'.
                    $bo_url.'">cliquant ici</a></b> ou depuis le menu Module > Store Commander</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_catalog=1">Ne plus afficher ce message</a></p>',
                'en' => '<p><b>Store Commander</b> is installed on your shop. Did you know that you can:</p>
                        <ul style="list-style-type: disc;padding-left:40px">
                        <li>Find and modify in mass all the information of your products</li>
                        <li>Export and re-import your entire catalog or a selection of products</li>
                        <li>Prepare sales and turbo speed promotions periods, controlling your margins</li>
                        <li>Improve all SEO criteria of each product sheet: meta tags, size of meta tags...</li>
                        <li>Identify and fix hundreds of problems with the Tools menu > FixMyPrestaShop</li>
                        </ul><br/>
                        <p>Start Store Commander to manage your catalog more efficiently, by <b><a href="'.$bo_url.'">clicking here</a></b> or from the Module > Store Commander menu</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_catalog=1">Do not display this message again</a></p>',
            ],
            'image' => [
                'fr' => '<p><b>Store Commander</b> est installé sur votre boutique.
                        <br/>Si la compression d\'image proposée par Store Commander est activée et que  vous regénérez vos miniatures, vous perdrez alors toute optimisation réalisée sur vos images produit.
                        <br/>Store Commander compressera de nouveau vos fichiers en utilisant les crédits restants.</p>
                        <p style="text-align:right"><a href="'.$hide_msg_url.
                    '&sc_no_msg_image=1">Ne plus afficher ce message</a></p>',
                'en' => '<p><b>Store Commander</b> is installed on your shop.</p>',
            ],
        ];
        if (array_key_exists($controller, $text) && array_key_exists($iso_lang, $text[$controller])) {
            $html = '<div id="sc_message" class="'.$class.'"'.$sc_message_style.'>'.$text[$controller][$iso_lang].
                '</a></div>';
            $html = json_encode($html);
        }

        return $html;
    }

    /**
     * @param string $pattern
     * @return mixed|string
     */
    private function getLastJsFileName(string $pattern)
    {
        $fs = new Symfony\Component\Finder\Finder();
        $jsDir = __DIR__.'/views/js/';
        $filesArray  = iterator_to_array($fs->files()->in($jsDir)->name($pattern)->sortByModifiedTime());
        return end($filesArray)->getRelativePathname();
    }

}
